<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class WPF_Logins_Public {

	/**
	 * Get things started
	 *
	 * @since 1.0
	 * @return void
	 */

	public function __construct() {

		add_action( 'init', array( $this, 'initialize_login_tracking' ) );

		add_action( 'wp_login', array( $this, 'user_login' ), 1 );

		add_action( 'wpf_logins_user_login', array( $this, 'do_redirects' ), 15 );

		// Cron job.
		add_action( 'wpf_logins_daily_check', array( $this, 'daily_check' ) );

	}

	/**
	 * Set up login tracking scheduled event if configured
	 *
	 * @access public
	 * @return void
	 */

	public function initialize_login_tracking() {

		$day_tracking = wpf_get_option( 'no_login_days' );
		$absence_tags = wpf_get_option( 'no_login_tags' );
		$next_event   = wp_next_scheduled( 'wpf_logins_daily_check' );

		if ( empty( $day_tracking ) || empty( $absence_tags ) ) {

			// Unschedule event if feature has been disabled.
			if ( ! empty( $next_event ) ) {
				wp_clear_scheduled_hook( 'wpf_logins_daily_check' );
			}
		} elseif ( empty( $next_event ) ) {

			// Schedule event if needed.
			wp_schedule_event( time(), 'daily', 'wpf_logins_daily_check' );

		}

		// Login counting (only run on users who registered after the plugin was installed).

		if ( is_user_logged_in() && 'daily' === wpf_get_option( 'login_count_method', 'daily' ) && ! wp_doing_ajax() && ! defined( 'REST_REQUEST' ) ) {

			if ( function_exists( 'current_user_switched' ) && current_user_switched() ) {
				return; // if we've switched into this account via User Switching.
			}

			$user_id = get_current_user_id();

			$last_login = get_user_meta( $user_id, 'wpf_last_login', true );

			if ( ! is_numeric( $last_login ) ) {
				$last_login = strtotime( $last_login );
			}

			// Check if its a new day.

			if ( gmdate( 'z', $last_login ) !== current_time( 'z' ) ) {

				// Update last login.
				update_user_meta( $user_id, 'wpf_last_login', current_time( 'U' ) );

				// Maybe remove stale account tags.

				$no_login_tags = wpf_get_option( 'no_login_tags', array() );

				if ( wpf_has_tag( $no_login_tags, $user_id ) ) {

					wp_fusion()->user->remove_tags( $no_login_tags, $user_id );

					delete_user_meta( $user_id, 'wpf_stale_account' );

					$recovered_login_tags = wpf_get_option( 'recovered_no_login_tags' );

					if ( ! empty( $recovered_login_tags ) ) {

						wp_fusion()->user->apply_tags( $recovered_login_tags, $user_id );

					}
				}

				// Update count.
				$login_count = $this->increment_login_count( $user_id );

				wp_fusion()->logger->add_source( 'wpf-logins' );

				$update_data = array(
					'wpf_last_login' => current_time( 'Y-m-d H:i:s' ),
					'login_count'    => $login_count,
				);

				wp_fusion()->user->push_user_meta( $user_id, $update_data );

				// Do redirects.
				$this->login_count_redirect( $user_id );

			}
		}

	}


	/**
	 * Finds last time user logged in, if user logs in for the first time apply tags
	 *
	 * @access public
	 * @return  void
	 */

	public function user_login( $user_login ) {

		if ( ! function_exists( 'wp_fusion' ) || empty( wp_fusion()->user ) ) {
			return;
		}

		$user = get_user_by( 'login', $user_login );

		// Do WPF login stuff early since this action might redirect and exit() and has a lower priority.
		wp_fusion()->user->login( $user_login );

		remove_action( 'wp_login', array( wp_fusion()->user, 'login' ), 10, 2 );

		$first_login_tags = wpf_get_option( 'first_login_apply_tags' );

		if ( ! empty( $first_login_tags ) ) {

			$last_user_login = get_user_meta( $user->ID, 'wpf_last_login', true );

			$userdata   = get_userdata( $user->ID );
			$registered = strtotime( $userdata->user_registered );

			$activation_date = get_option( 'wpf_logins_activation_date', current_time( 'U' ) );

			if ( empty( $last_user_login ) && $registered > $activation_date ) {
				wp_fusion()->user->apply_tags( $first_login_tags, $user->ID );
			}
		}

		$no_login_tags = wpf_get_option( 'no_login_tags' );

		if ( ! empty( $no_login_tags ) ) {

			$user_tags = wp_fusion()->user->get_tags( $user->ID );

			// If user has a stale account tag, remove it.
			$result = array_intersect( $no_login_tags, $user_tags );

			if ( ! empty( $result ) ) {

				wp_fusion()->user->remove_tags( $no_login_tags, $user->ID );

				delete_user_meta( $user->ID, 'wpf_stale_account' );

				$recovered_login_tags = wpf_get_option( 'recovered_no_login_tags' );

				if ( ! empty( $recovered_login_tags ) ) {

					wp_fusion()->user->apply_tags( $recovered_login_tags, $user->ID );

				}
			}
		}

		// Update count.
		if ( 'true' === wpf_get_option( 'login_count_method' ) ) {

			update_user_meta( $user->ID, 'wpf_last_login', current_time( 'U' ) );

			// Update count.
			$login_count = $this->increment_login_count( $user->ID );

			wp_fusion()->logger->add_source( 'wpf-logins' );

			$update_data = array(
				'wpf_last_login' => current_time( 'Y-m-d H:i:s' ),
				'login_count'    => $login_count,
			);

			wp_fusion()->user->push_user_meta( $user->ID, $update_data );

		}

		// Do redirects.
		do_action( 'wpf_logins_user_login', $user->ID );

	}

	/**
	 * Handles redirects on login
	 *
	 * @access public
	 * @return  void
	 */

	public function do_redirects( $user_id ) {

		if ( user_can( $user_id, 'manage_options' ) || wp_doing_ajax() || defined( 'REST_REQUEST' ) ) {
			return;
		}

		if ( 'true' === wpf_get_option( 'login_count_method', 'true' ) ) {

			$this->login_count_redirect( $user_id );

		}

		$this->user_redirect( $user_id );

	}


	/**
	 * Increments the login count for a user
	 *
	 * @access public
	 * @return void
	 */

	public function increment_login_count( $user_id ) {

		$login_count = get_user_meta( $user_id, 'login_count', true );

		if ( empty( $login_count ) ) {
			$login_count = 0;
		}

		$login_count++;

		update_user_meta( $user_id, 'login_count', $login_count );

		return $login_count;

	}

	/**
	 * Gets the redirect URL (login redirect)
	 *
	 * @access public
	 * @return string Redirect URL
	 */

	public function get_user_redirect( $user_id ) {

		// Login redirect rules.
		$redirect_rules = wpf_get_option( 'login_redirect_rules' );

		if ( empty( $redirect_rules ) ) {
			return false;
		}

		$user_tags = wp_fusion()->user->get_tags( $user_id );

		foreach ( $redirect_rules as $rule ) {

			if ( empty( $rule['page'] ) && empty( $rule['redirect_url'] ) ) {
				continue;
			}

			if ( ! isset( $rule['tags'] ) ) {
				$rule['tags'] = array();
			}

			$result = array_intersect( $user_tags, $rule['tags'] );

			if ( ! empty( $result ) || empty( $rule['tags'] ) ) {

				if ( ! empty( $rule['page'] ) && empty( $rule['redirect_url'] ) ) {

					return ( intval( $rule['page'] ) !== 0 ? get_permalink( $rule['page'] ) : $rule['page'] );

				} elseif ( ! empty( $rule['redirect_url'] ) ) {
					return $rule['redirect_url'];
				}
			}
		}

		return false;

	}

	/**
	 * Redirects users based on tags
	 *
	 * @access public
	 * @return void
	 */

	public function user_redirect( $user_id = false ) {

		if ( false == $user_id ) {
			$user_id = get_current_user_id();
		}

		$redirect = $this->get_user_redirect( $user_id );

		if ( false !== $redirect ) {

			wp_redirect( $redirect );
			exit;

		}

	}

	/**
	 * Gets the redirect URL based on login count
	 *
	 * @access public
	 * @return string Redirect URL
	 */

	public function get_login_count_redirect( $user_id ) {

		// Login redirect rules.
		$redirect_rules = wpf_get_option( 'login_count_rules' );

		if ( empty( $redirect_rules ) ) {
			return false;
		}

		$login_count = get_user_meta( $user_id, 'login_count', true );

		foreach ( $redirect_rules as $rule ) {

			if ( empty( $rule['count'] ) ) {
				continue;
			}

			// Check tags first.
			if ( ! empty( $rule['tags'] ) ) {

				$user_tags = wp_fusion()->user->get_tags( $user_id );

				if ( empty( $user_tags ) ) {
					continue;
				}

				$result = array_intersect( $user_tags, $rule['tags'] );

				if ( empty( $result ) ) {
					continue;
				}
			}

			if ( $rule['type'] == 'on' ) {

				if ( $rule['count'] == $login_count ) {

					if ( ! empty( $rule['apply_tags'] ) ) {

						wp_fusion()->user->apply_tags( $rule['apply_tags'], $user_id );

					}

					if ( ! empty( $rule['page'] ) ) {

						return ( intval( $rule['page'] ) !== 0 ? get_permalink( $rule['page'] ) : $rule['page'] );

					}
				}
			} elseif ( $rule['type'] == 'every' ) {

				if ( $login_count % $rule['count'] == 0 ) {

					if ( ! empty( $rule['apply_tags'] ) ) {

						wp_fusion()->user->apply_tags( $rule['apply_tags'], $user_id );

					}

					if ( ! empty( $rule['page'] ) ) {

						return ( intval( $rule['page'] ) !== 0 ? get_permalink( $rule['page'] ) : $rule['page'] );

					}
				}
			}
		}

		return false;

	}


	/**
	 * Possibly redirects users based on login count
	 *
	 * @access public
	 * @return void
	 */

	public function login_count_redirect( $user_id = false ) {

		if ( false == $user_id ) {
			$user_id = get_current_user_id();
		}

		$redirect = $this->get_login_count_redirect( $user_id );

		if ( false !== $redirect ) {

			wp_redirect( $redirect );
			exit;

		}

	}


	/**
	 * Applies tags if user has not logged in for a certain amount of days
	 *
	 * @access public
	 * @return  void
	 */

	public function daily_check() {

		$no_login_days = wpf_get_option( 'no_login_days' );
		$no_login_time = $no_login_days * DAY_IN_SECONDS;
		$compare       = current_time( 'U' ) - $no_login_time;
		$compare_start = $compare - ( DAY_IN_SECONDS * 2); // two days in case the cron doesn't run exactly on time.

		$args = array(
			'fields'     => 'ID',
			'meta_query' => array(
				array(
					'key'     => 'wpf_last_login',
					'compare' => 'EXISTS',
				),
				array(
					'key'     => 'wpf_last_login',
					'value'   => $compare,
					'compare' => '<',
				),
				array(
					'key'     => 'wpf_last_login',
					'value'   => $compare_start,
					'compare' => '>',
				),
				array(
					'key'     => WPF_CONTACT_ID_META_KEY,
					'compare' => 'EXISTS',
				),
				array(
					'key'     => 'wpf_stale_account',
					'compare' => 'NOT EXISTS',
				),
			),
		);

		$users = get_users( $args );

		if ( empty( $users ) ) {
			return;
		}

		wpf_log( 'info', 0, 'Processing stale account tagging for ' . count( $users ) . ' user(s).', array( 'source' => 'wpf-logins' ) );
 
		$absence_tags = wpf_get_option( 'no_login_tags' );

		foreach ( $users as $user_id ) {

			$last_login = get_user_meta( $user_id, 'wpf_last_login', true );

			if ( empty( $last_login ) || ! is_numeric( $last_login ) ) {
				continue;
			}

			// Quit early if user already has the tag.

			if ( wpf_has_tag( $absence_tags, $user_id ) ) {
				continue;
			}

			wpf_log( 'info', $user_id, 'User hasn\'t logged in for ' . $no_login_days . ' days.', array( 'source' => 'wpf-logins' ) );

			wp_fusion()->user->apply_tags( $absence_tags, $user_id );

			update_user_meta( $user_id, 'wpf_stale_account', true );

			// Maybe remove the recovered account tags.

			$recovered_login_tags = wpf_get_option( 'recovered_no_login_tags' );

			if ( ! empty( $recovered_login_tags ) ) {

				wp_fusion()->user->remove_tags( $recovered_login_tags, $user_id );

			}
		}

	}

}
