<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class WPF_Logins_Compat {

	/**
	 * Get things started
	 *
	 * @since 1.0
	 * @return void
	 */

	public function __construct() {

		// AJAX login bypasses.
		add_action( 'init', array( $this, 'maybe_doing_ajax_login' ), 1 );

		// PilotPress overrides.
		add_action( 'pilotpress_post_user_login', array( wp_fusion_logins()->public, 'user_redirect' ) );

		// Ultimate Member.
		add_action( 'um_on_login_before_redirect', array( $this, 'um_user_redirect' ), 10 );

		// WooCommerce login after checkout.
		add_action( 'template_redirect', array( $this, 'woocommerce_redirect' ) );

		add_action( 'wp_login', array( $this, 'woocommerce_maybe_bypass_redirects' ), 0 );

		add_action( 'set_logged_in_cookie', array( $this, 'woocommerce_registration' ), 10, 4 );

		// Login with Ajax.
		add_filter( 'lwa_ajax_login', array( $this, 'lwa_login' ) );

	}

	/**
	 * Suppress redirects during AJAX login
	 *
	 * @access public
	 * @return void
	 */

	public function maybe_doing_ajax_login() {

		$request_params = array(
			'login-with-ajax',
		);

		$request_params = apply_filters( 'wpf_logins_bypass_request_params', $request_params );

		foreach ( $request_params as $param ) {

			if ( isset( $_REQUEST[ $param ] ) ) {

				remove_action( 'wpf_logins_user_login', array( wp_fusion_logins()->public, 'do_redirects' ) );

			}
		}
	}

	/**
	 * Redirects users based on tags (Ultimate Member)
	 *
	 * @access public
	 * @return string
	 */

	public function um_user_redirect( $user_id ) {

		wp_fusion_logins()->public->user_redirect( $user_id );

	}

	/**
	 * WooCommerce redirect after login at purchase
	 *
	 * @access public
	 * @return array Redirect
	 */

	public function woocommerce_redirect() {

		if ( ! function_exists( 'is_wc_endpoint_url' ) || ! is_wc_endpoint_url( 'order-received' ) || ! is_user_logged_in() ) {
			return;
		}

		if ( wpf_get_option( 'login_woo_redirect' ) ) {

			// Do redirects.
			do_action( 'wpf_logins_user_login', wpf_get_current_user_id() );
		}

	}

	/**
	 * Bypass the login redirects on the checkout / account login forms
	 *
	 * @access public
	 * @return void
	 */

	public function woocommerce_maybe_bypass_redirects() {

		if ( isset( $_POST['woocommerce-login-nonce'] ) ) {

			remove_action( 'wpf_logins_user_login', array( wp_fusion_logins()->public, 'do_redirects' ), 15 );

		}

	}

	/**
	 * The WooCommerce auto-login after checkout bypasses the wp_login hook so this
	 * will ensure any tags / data is correctly synced.
	 *
	 * @since 1.3.3
	 *
	 * @param string $logged_in_cookie The logged-in cookie value.
	 * @param int    $expire           The time the login grace period expires as a UNIX timestamp.
	 *                                 Default is 12 hours past the cookie's expiration time.
	 * @param int    $expiration       The time when the logged-in authentication cookie expires as a UNIX timestamp.
	 *                                 Default is 14 days from now.
	 * @param int    $user_id          User ID.
	 */
	public function woocommerce_registration( $logged_in_cookie, $expire, $expiration, $user_id ) {

		if ( doing_action( 'wc_ajax_checkout' ) ) {

			$user = get_user_by( 'id', $user_id );

			wp_fusion_logins()->public->user_login( $user->user_login );

		}
	}

	/**
	 * Login with AJAX compat
	 *
	 * @access public
	 * @return array Return
	 */

	public function lwa_login( $return ) {

		if ( true == $return['result'] ) {

			$user = get_user_by( 'login', $_REQUEST['log'] );

			if ( empty( $user ) ) {
				$user = get_user_by( 'email', $_REQUEST['log'] );
			}

			$redirect = wp_fusion_logins()->public->get_user_redirect( $user->ID );

			if ( wp_fusion()->settings->get( 'login_count_method', 'true' ) == 'true' ) {

				$alt_redirect = wp_fusion_logins()->public->get_login_count_redirect( $user->ID );

				if ( false !== $alt_redirect ) {
					$redirect = $alt_redirect;
				}
			}

			if ( false !== $redirect ) {
				$return['redirect'] = $redirect;
			}
		}

		return $return;

	}

}
