<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class WPF_Logins_Admin {

	/**
	 * Get things started
	 *
	 * @since 1.0
	 * @return void
	 */

	public function __construct() {

		add_action( 'init', array( $this, 'set_install_date' ) );

		// Add tab and settings
		add_filter( 'wpf_configure_sections', array( $this, 'configure_sections' ), 10, 2 );
		add_filter( 'wpf_configure_settings', array( $this, 'register_settings' ), 15, 2 );

		// Users columns
		add_filter( 'manage_users_columns', array( $this, 'manage_users_columns' ) );
		add_filter( 'manage_users_custom_column', array( $this, 'manage_users_custom_column' ), 10, 3 );
		add_filter( 'manage_users_sortable_columns', array( $this, 'manage_users_sortable_columns' ) );
		add_filter( 'users_list_table_query_args', array( $this, 'users_list_table_query_args' ) );

		// Scripts and styles
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

		// Custom repeater setting output
		add_action( 'show_field_login_redirect_rules', array( $this, 'show_field_login_redirect_rules' ), 10, 2 );
		add_action( 'show_field_login_count_rules', array( $this, 'show_field_login_count_rules' ), 10, 2 );

		// Save settings
		add_action( 'wpf_settings_page_init', array( $this, 'save_options' ) );

		add_filter( 'wpf_meta_field_groups', array( $this, 'add_meta_field_group' ), 10 );
		add_filter( 'wpf_meta_fields', array( $this, 'prepare_meta_fields' ), 10 );

		// User profile
		add_action( 'wpf_user_profile_after_table_rows', array( $this, 'user_profile_table' ) );

	}

	/**
	 * Add Last Login field to contact fields list
	 *
	 * @access  public
	 * @since   1.0
	 * @return  array Field Groups
	 */

	public function add_meta_field_group( $field_groups ) {

		$field_groups['logins'] = array(
			'title'  => 'WP Fusion Logins',
			'url'    => 'https://wpfusion.com/documentation/logins/overview/#syncing-meta-fields',
			'fields' => array(),
		);

		return $field_groups;

	}

	/**
	 * Adds Last Login meta field to WPF contact fields list
	 *
	 * @access  public
	 * @return  array Meta Fields
	 */

	public function prepare_meta_fields( $meta_fields ) {

		$meta_fields['wpf_last_login'] = array(
			'label'  => 'Last Login',
			'type'   => 'date',
			'group'  => 'logins',
			'pseudo' => true,
		);
		$meta_fields['login_count']    = array(
			'label'  => 'Login Count',
			'type'   => 'integer',
			'group'  => 'logins',
			'pseudo' => true,
		);

		return $meta_fields;

	}


	/**
	 * Add Last Login column to users list
	 *
	 * @access public
	 * @return array Columns
	 */

	public function manage_users_columns( $columns ) {

		$columns['wpf_last_login'] = __( 'Last Login', 'wp-fusion-logins' );

		return $columns;

	}

	/**
	 * Last Login column content
	 *
	 * @access public
	 * @return string Last login
	 */

	public function manage_users_custom_column( $val, $column_name, $user_id ) {

		if ( 'wpf_last_login' == $column_name ) {

			$last_login = get_user_meta( $user_id, 'wpf_last_login', true );

			if ( empty( $last_login ) ) {

				return '-';

			} else {

				return date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), intval( $last_login ) );

			}
		}

		return $val;

	}

	/**
	 * Make last login sortable
	 *
	 * @access public
	 * @return array Sortable Columns
	 */

	public function manage_users_sortable_columns( $sortable_columns ) {

		$sortable_columns['wpf_last_login'] = 'wpf-lastlogin';

		return $sortable_columns;

	}

	/**
	 * Sort users by last login
	 *
	 * @access public
	 * @return array Query Args
	 */

	public function users_list_table_query_args( $args ) {

		if ( isset( $args['orderby'] ) && $args['orderby'] == 'wpf-lastlogin' ) {

			$args['meta_query'] = array(
				'relation' => 'OR',
				array(
					'key'     => 'wpf_last_login',
					'compare' => 'EXISTS',
				),
				array(
					'key'     => 'wpf_last_login',
					'compare' => 'NOT EXISTS',
				),
			);

			$args['orderby'] = 'meta_value_num';

		}

		return $args;

	}


	/**
	 * Shows login count on user profile
	 *
	 * @access  public
	 * @return  mixed HTML Output
	 */

	public function user_profile_table( $user ) {

		?>
			<tr>
				<th><label><?php _e( 'Login Count', 'wp-fusion-logins' ); ?></label></th>
				<td><?php echo get_user_meta( $user->ID, 'login_count', true ); ?></td>
			</tr>
			<tr>
				<th><label><?php _e( 'Last Login', 'wp-fusion-logins' ); ?></label></th>
				<td>
				<?php

				$date = get_user_meta( $user->ID, 'wpf_last_login', true );

				if ( empty( $date ) ) {
					echo 'none';
				} else {
					echo date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), intval( $date ) );
				}

				?>
				</td>
			</tr>

		<?php

	}

	/**
	 * Stores the date of activation so "first login" tags aren't applied to existing users
	 *
	 * @access public
	 * @return void
	 */

	public function set_install_date() {

		$activation_date = get_option( 'wpf_logins_activation_date' );

		if ( ! empty( $activation_date ) ) {
			return;
		}

		update_option( 'wpf_logins_activation_date', current_time( 'U' ) );

	}

	/**
	 * Adds Addons tab if not already present
	 *
	 * @access public
	 * @return void
	 */

	public function configure_sections( $page, $options ) {

		$page['sections'] = wp_fusion()->settings->insert_setting_before( 'import', $page['sections'], array( 'logins' => __( 'Logins', 'wp-fusion-logins' ) ) );

		return $page;

	}


	/**
	 * Enqueue admin scripts
	 *
	 * @since 1.0
	 * @return void
	 */

	public function enqueue_scripts() {

		$screen = get_current_screen();

		if ( $screen->id == 'settings_page_wpf-settings' ) {

			wp_enqueue_script( 'dragula', WPF_LOGINS_DIR_URL . 'assets/dragula/dragula.min.js' );
			wp_enqueue_style( 'dragula', WPF_LOGINS_DIR_URL . 'assets/dragula/dragula.css' );

			wp_enqueue_script( 'wpf-logins-admin', WPF_LOGINS_DIR_URL . 'assets/wpf-logins-admin.js', array( 'jquery', 'dragula' ), WPF_LOGINS_VERSION, true );
			wp_enqueue_style( 'wpf-logins-admin', WPF_LOGINS_DIR_URL . 'assets/wpf-logins-admin.css' );

		}

	}

	/**
	 * Add fields to settings page
	 *
	 * @access public
	 * @return array Settings
	 */

	public function register_settings( $settings, $options ) {

		$settings['login_header'] = array(
			'desc'    => '<p>' . sprintf( __( 'For more information on these settings, %1$ssee our documentation%2$s.', 'wp-fusion' ), '<a href="https://wpfusion.com/documentation/logins/overview/" target="_blank">', '</a>' ) . '</p>',
			'type'    => 'heading',
			'section' => 'logins',
		);

		$settings['first_login_apply_tags'] = array(
			'title'   => __( 'Apply Tags - First Login', 'wp-fusion-logins' ),
			'desc'    => __( 'Apply these tags when a user first logs in.', 'wp-fusion-logins' ),
			'std'     => array(),
			'type'    => 'assign_tags',
			'section' => 'logins',
		);

		$settings['no_login_tags'] = array(
			'title'   => __( 'Apply Tags - Stale Accounts', 'wp-fusion-logins' ),
			'desc'    => __( 'Apply these tags when a user has not logged in for the specified period of time. Tags will be removed on next login.', 'wp-fusion-logins' ),
			'std'     => array(),
			'type'    => 'assign_tags',
			'section' => 'logins',
		);

		$settings['no_login_days'] = array(
			'title'    => __( 'Number of Days', 'wp-fusion-logins' ),
			'std'      => 30,
			'type'     => 'number',
			'section'  => 'logins',
			'min'      => 0,
			'max'      => 1000,
			'disabled' => false,
			'class'    => false,
		);

		$settings['recovered_no_login_tags'] = array(
			'title'   => __( 'Apply Tags - Recovered Accounts', 'wp-fusion-logins' ),
			'desc'    => __( 'Apply these tags when a user tagged as stale (above) logs in again. Will be removed if account goes stale again.', 'wp-fusion-logins' ),
			'std'     => array(),
			'type'    => 'assign_tags',
			'section' => 'logins',
		);

		// Custom field output
		$settings['login_count_rules'] = array(
			'title'   => __( 'Login Count Redirect Rules', 'wp-fusion-logins' ),
			'std'     => array(),
			'type'    => 'repeater',
			'section' => 'logins',
		);

		// Custom field output
		$settings['login_redirect_rules'] = array(
			'title'   => __( 'Login Redirect Rules', 'wp-fusion-logins' ),
			'std'     => array(),
			'type'    => 'repeater',
			'section' => 'logins',
		);

		// Custom field output
		$settings['login_count_method'] = array(
			'title'   => __( 'Login Count Method', 'wp-fusion-logins' ),
			'std'     => 'daily',
			'type'    => 'radio',
			'section' => 'logins',
			'choices' => array(
				'daily' => __( 'Once a day on site visit', 'wp-fusion-logins' ),
				'true'  => __( 'When a user actually logs in', 'wp-fusion-logins' ),
			),
		);

		if ( class_exists( 'WooCommerce' ) ) {

			$settings['login_integrations_header'] = array(
				'title'   => __( 'Logins Integrations', 'wp-fusion-logins' ),
				'std'     => 0,
				'type'    => 'heading',
				'section' => 'logins',
			);

			$settings['login_woo_redirect'] = array(
				'title'   => __( 'WooCommerce Redirects', 'wp-fusion-logins' ),
				'desc'    => __( 'Run redirect rules when a logged in user reaches the Order Received page in WooCommerce.', 'wp-fusion-logins' ),
				'type'    => 'checkbox',
				'section' => 'logins',
			);

		}

		return $settings;

	}


	/**
	 * Formats Repeater Field
	 *
	 * @access public
	 * @return array Settings
	 */

	public function show_field_login_redirect_rules( $id, $field ) {

		$rules = wp_fusion()->settings->get( 'login_redirect_rules' );

		if ( empty( $rules ) ) {
			$rules = array();
		}

		foreach ( $rules as $i => $rule ) {

			if ( empty( $rule['page'] ) && empty( $rule['apply_tags'] ) && empty( $rule['redirect_url'] ) ) {
				unset( $rules[ $i ] );
			}
		}

		$rules[] = array(
			'tags'         => array(),
			'page'         => '',
			'redirect_url' => '',
		);

		$rules = array_values( $rules );

		?>
		<div id="wpf-repeater-rules" class="wpf-repeater dragula-container">
			<?php

			$class = 'clone';

			foreach ( $rules as $i => $rule ) :

				if ( empty( $rule['page'] ) && empty( $rule['apply_tags'] ) && empty( $rule['redirect_url'] ) ) {
					$class .= ' repeater';
				}

				?>

				<div class="<?php echo $class; ?>" data-num="<?php echo $i; ?>">

				<div class="bars">
					<span></span>
					<span></span>
					<span></span>
				</div>

				<?php

				if ( ! isset( $rule['tags'] ) ) { // This should be fixed somewhere better but I'm lazy
					$rule['tags'] = array();
				}

				$args = array(
					'setting'     => $rule['tags'],
					'meta_name'   => "login_redirect_rules[{$i}][tags]",
					'placeholder' => __( 'Select tags (leave blank for any)', 'wp-fusion-logins' ),
				);

				wpf_render_tag_multiselect( $args );

				echo '<span class="raquo">&raquo;</span>';
				echo '<select class="select4-select-page select4-allow-adding" data-placeholder="Redirect To Page or URL" name="login_redirect_rules[' . $i . '][page]">';

				echo '<option></option>';

				if ( ! empty( $rule['page'] ) ) {
					echo '<option value="' . $rule['page'] . '" selected>' . ( intval( $rule['page'] ) !== 0 ? get_the_title( $rule['page'] ) : $rule['page'] ) . '</option>';
				} else {
					echo '<option value="' . $rule['redirect_url'] . '" selected>' . $rule['redirect_url'] . '</option>';
				}

				echo '</select>';

				?>

				<div class="minus">
					<svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="12" height="14" viewBox="0 0 12 14">
					<path d="M9.5 7.5v-1q0-0.203-0.148-0.352t-0.352-0.148h-6q-0.203 0-0.352 0.148t-0.148 0.352v1q0 0.203 0.148 0.352t0.352 0.148h6q0.203 0 0.352-0.148t0.148-0.352zM12 7q0 1.633-0.805 3.012t-2.184 2.184-3.012 0.805-3.012-0.805-2.184-2.184-0.805-3.012 0.805-3.012 2.184-2.184 3.012-0.805 3.012 0.805 2.184 2.184 0.805 3.012z"></path>
					</svg>
				</div>
				</div>

			<?php endforeach; ?>
		</div>

		<a id="logins_redirect_add_new" href="#addons"><div class="inline_text"><div class="inline"><svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="12" height="14" viewBox="0 0 12 14">
		<path d="M9.5 7.5v-1q0-0.203-0.148-0.352t-0.352-0.148h-2v-2q0-0.203-0.148-0.352t-0.352-0.148h-1q-0.203 0-0.352 0.148t-0.148 0.352v2h-2q-0.203 0-0.352 0.148t-0.148 0.352v1q0 0.203 0.148 0.352t0.352 0.148h2v2q0 0.203 0.148 0.352t0.352 0.148h1q0.203 0 0.352-0.148t0.148-0.352v-2h2q0.203 0 0.352-0.148t0.148-0.352zM12 7q0 1.633-0.805 3.012t-2.184 2.184-3.012 0.805-3.012-0.805-2.184-2.184-0.805-3.012 0.805-3.012 2.184-2.184 3.012-0.805 3.012 0.805 2.184 2.184 0.805 3.012z"></path>
		</svg></div>Add new redirect rule.</div></a>

		<span class="description">Rules are prioritized from top to bottom. For each tag condition, you can select a page on your site or external URL to redirect the user to on login.</span>

		<?php
	}



	/**
	 * Formats Repeater Field
	 *
	 * @access public
	 * @return array Settings
	 */

	public function show_field_login_count_rules( $id, $field ) {

		$rules = wp_fusion()->settings->get( 'login_count_rules' );

		if ( empty( $rules ) ) {
			$rules = array();
		}

		$defaults = array(
			'count'      => '1',
			'type'       => 'on',
			'page'       => '',
			'tags'       => array(),
			'apply_tags' => array(),
		);

		foreach ( $rules as $i => $rule ) {

			if ( empty( $rule['page'] ) && empty( $rule['apply_tags'] ) ) {
				unset( $rules[ $i ] );
			} else {
				$rules[ $i ] = array_merge( $defaults, $rules[ $i ] );
			}
		}

		$rules[] = $defaults;

		$rules = array_values( $rules );

		?>
		<div id="wpf-repeater-count-rules" class="wpf-repeater">
		<?php

		$class = 'clone';

		foreach ( $rules as $i => $rule ) :

			if ( empty( $rule['page'] ) && empty( $rule['apply_tags'] ) ) {
				$class .= ' repeater';
			}

			?>

				<div class="<?php echo $class; ?>" data-num="<?php echo $i; ?>">

					<select class="select4" name="login_count_rules[<?php echo $i; ?>][type]">
						<option value="on" <?php echo selected( 'on', $rule['type'], false ); ?>>On the Nth login</option>
						<option value="every" <?php echo selected( 'every', $rule['type'], false ); ?>>Every N logins</option>
					</select>

					<input class="form-control" type="number" name="login_count_rules[<?php echo $i; ?>][count]" min="1" value="<?php echo $rule['count']; ?>" ></input>

					<?php

					$args = array(
						'setting'     => $rule['tags'],
						'meta_name'   => "login_count_rules[{$i}][tags]",
						'placeholder' => __( 'Required tags (optional)', 'wp-fusion-logins' ),
					);

					wpf_render_tag_multiselect( $args );

					echo '<span class="raquo">&raquo;</span><select class="select4-select-page select4-allow-adding" data-placeholder="Redirect To Page or URL" name="login_count_rules[' . $i . '][page]">';

					echo '<option></option>';

					if ( ! empty( $rule['page'] ) ) {
						echo '<option value="' . $rule['page'] . '" selected>' . ( intval( $rule['page'] ) !== 0 ? get_the_title( $rule['page'] ) : $rule['page'] ) . '</option>';
					}

					echo '</select>';

					$args = array(
						'setting'     => $rule['apply_tags'],
						'meta_name'   => "login_count_rules[{$i}][apply_tags]",
						'placeholder' => __( 'Apply tags', 'wp-fusion-logins' ),
					);

					wpf_render_tag_multiselect( $args );

					?>
				<div class="minus">
					<svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="12" height="14" viewBox="0 0 12 14">
					<path d="M9.5 7.5v-1q0-0.203-0.148-0.352t-0.352-0.148h-6q-0.203 0-0.352 0.148t-0.148 0.352v1q0 0.203 0.148 0.352t0.352 0.148h6q0.203 0 0.352-0.148t0.148-0.352zM12 7q0 1.633-0.805 3.012t-2.184 2.184-3.012 0.805-3.012-0.805-2.184-2.184-0.805-3.012 0.805-3.012 2.184-2.184 3.012-0.805 3.012 0.805 2.184 2.184 0.805 3.012z"></path>
					</svg>
				</div>
				</div>

			<?php endforeach; ?>
		</div>

		<a id="logins_count_add_new" href="#addons"><div class="inline_text"><div class="inline"><svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="12" height="14" viewBox="0 0 12 14">
		<path d="M9.5 7.5v-1q0-0.203-0.148-0.352t-0.352-0.148h-2v-2q0-0.203-0.148-0.352t-0.352-0.148h-1q-0.203 0-0.352 0.148t-0.148 0.352v2h-2q-0.203 0-0.352 0.148t-0.148 0.352v1q0 0.203 0.148 0.352t0.352 0.148h2v2q0 0.203 0.148 0.352t0.352 0.148h1q0.203 0 0.352-0.148t0.148-0.352v-2h2q0.203 0 0.352-0.148t0.148-0.352zM12 7q0 1.633-0.805 3.012t-2.184 2.184-3.012 0.805-3.012-0.805-2.184-2.184-0.805-3.012 0.805-3.012 2.184-2.184 3.012-0.805 3.012 0.805 2.184 2.184 0.805 3.012z"></path>
		</svg></div>Add new count rule.</div></a>
		
		<span class="description">For each condition you can specify a redirect to be triggered and / or a tag to be applied.</span>

		<?php
	}


	/**
	 * Formats Save Settings
	 *
	 * @access public
	 * @return void
	 */

	public function save_options() {

		if ( isset( $_POST['login_redirect_rules'] ) ) {

			$_POST['wpf_options']['login_redirect_rules'] = $_POST['login_redirect_rules'];

		}

		if ( isset( $_POST['login_count_rules'] ) ) {

			$_POST['wpf_options']['login_count_rules'] = $_POST['login_count_rules'];

		}

	}


}

new WPF_Logins_Admin();
