<?php

/**
 * Plugin Name: WP Fusion - Event Tracking Addon
 * Description: Track events in your CRM.
 * Plugin URI: https://wpfusion.com/
 * Version: 1.3.1
 * Author: Very Good Plugins
 * Author URI: https://verygoodplugins.com/
 * Text Domain: wp-fusion-event-tracking
 */

/**
 * @copyright Copyright (c) 2022. All rights reserved.
 *
 * @license   Released under the GPL license http://www.opensource.org/licenses/gpl-license.php
 *
 * **********************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * **********************************************************************
 */


// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'WPF_EVENT_TRACKING_VERSION', '1.3.1' );


/**
 * Class WP_Fusion_Event_Tracking
 *
 * @since 1.0.0
 */
final class WP_Fusion_Event_Tracking {

	/** Singleton *************************************************************/

	/**
	 * The one true WP_Fusion_Event_Tracking.
	 *
	 * @var WP_Fusion_Event_Tracking
	 * @since 1.0.0
	 */
	private static $instance;

	/**
	 * Allows interfacing with the main class.
	 *
	 * @var WPF_Event_Tracking_Public
	 * @since 1.0.0
	 */
	public $public;

	/**
	 * Allows interfacing with the admin class.
	 *
	 * @var WPF_Event_Tracking_Admin
	 * @since 1.1.3
	 */
	public $admin;

	/**
	 * Allows interfacing with integrations.
	 *
	 * @var Integrations.
	 * @since 1.0.0
	 */
	public $integrations;


	/**
	 * Main WP_Fusion_Event_Tracking Instance
	 *
	 * Insures that only one instance of WP_Fusion_Event_Tracking exists in
	 * memory at any one time. Also prevents needing to define globals all over
	 * the place.
	 *
	 * @since  1.0.0
	 * @static var array $instance
	 *
	 * @return WP_Fusion_Event_Tracking The one true WP_Fusion_Event_Tracking
	 */
	public static function instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof WP_Fusion_Event_Tracking ) ) {

			self::$instance = new WP_Fusion_Event_Tracking();

			self::$instance->setup_constants();

			if ( ! is_wp_error( self::$instance->check_install() ) ) {

				self::$instance->includes();

				// Integration modules are stored here for easy access, for
				// example wp_fusion_event_tracking()->integrations->{'woocommerce'}->get_events().

				self::$instance->integrations = new stdClass();
				self::$instance->integrations_includes();

				self::$instance->public = new WPF_Event_Tracking_Public();

				if ( is_admin() ) {
					self::$instance->admin = new WPF_Event_Tracking_Admin();
				}

				add_action( 'init', array( self::$instance, 'updater' ) );

			} else {
				add_action( 'admin_notices', array( self::$instance, 'admin_notices' ) );
			}
		}

		return self::$instance;
	}

	/**
	 * Throw error on object clone.
	 *
	 * The whole idea of the singleton design pattern is that there is a single
	 * object therefore, we don't want the object to be cloned.
	 *
	 * @access protected
	 *
	 * @since  1.0.0.
	 */
	public function __clone() {
		// Cloning instances of the class is forbidden.
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Cheatin&#8217; huh?', 'wp-fusion-event-tracking' ), '1.0.0' );
	}

	/**
	 * Disable unserializing of the class.
	 *
	 * @access protected
	 *
	 * @since  1.0.0
	 */
	public function __wakeup() {
		// Unserializing instances of the class is forbidden.
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Cheatin&#8217; huh?', 'wp-fusion-event-tracking' ), '1.0.0' );
	}

	/**
	 * If the method doesn't exist here, call it from the public class.
	 *
	 * @since  1.0.0
	 *
	 * @param  string $name      The method name.
	 * @param  array  $arguments The arguments.
	 * @return mxied  The returned value.
	 */
	public function __call( $name, $arguments ) {

		if ( ! method_exists( self::$instance, $name ) && method_exists( self::$instance->public, $name ) ) {
			return call_user_func_array( array( self::$instance->public, $name ), $arguments );
		}

	}

	/**
	 * Setup plugin constants.
	 *
	 * @access private
	 *
	 * @since  1.0.0
	 */
	private function setup_constants() {

		if ( ! defined( 'WPF_EVENT_TRACKING_DIR_PATH' ) ) {
			define( 'WPF_EVENT_TRACKING_DIR_PATH', plugin_dir_path( __FILE__ ) );
		}

		if ( ! defined( 'WPF_EVENT_TRACKING_PLUGIN_PATH' ) ) {
			define( 'WPF_EVENT_TRACKING_PLUGIN_PATH', plugin_basename( __FILE__ ) );
		}

		if ( ! defined( 'WPF_EVENT_TRACKING_DIR_URL' ) ) {
			define( 'WPF_EVENT_TRACKING_DIR_URL', plugin_dir_url( __FILE__ ) );
		}

		if ( ! defined( 'WPF_EVENT_TRACKING_ENDPOINT' ) ) {
			define( 'WPF_EVENT_TRACKING_ENDPOINT', 'wpfd' );
		}

	}


	/**
	 * Include required files.
	 *
	 * @access private
	 *
	 * @since  1.0.0
	 */
	private function includes() {

		require_once WPF_EVENT_TRACKING_DIR_PATH . 'includes/class-wpf-event-tracking-public.php';

		if ( is_admin() ) {
			require_once WPF_EVENT_TRACKING_DIR_PATH . 'includes/admin/class-wpf-event-tracking-admin.php';
			require_once WPF_EVENT_TRACKING_DIR_PATH . 'includes/admin/admin-functions.php';
		}

	}

	/**
	 * Include plugin integration classes.
	 *
	 * @access private
	 *
	 * @since  1.0.0
	 */
	private function integrations_includes() {

		// Autoload integrations.
		require_once WPF_EVENT_TRACKING_DIR_PATH . 'includes/integrations/class-integrations-base.php';

		foreach ( wp_fusion()->integrations as $integration ) {

			if ( file_exists( WPF_EVENT_TRACKING_DIR_PATH . 'includes/integrations/' . $integration->slug . '/class-' . $integration->slug . '.php' ) ) {
				include WPF_EVENT_TRACKING_DIR_PATH . 'includes/integrations/' . $integration->slug . '/class-' . $integration->slug . '.php';
			}
		}

		// Extend integrations for integrations that does not have files in WPF integrations folder.

		$exteded_integrations = array(
			'presto-player' => 'PrestoPlayer\Core',
		);

		foreach ( $exteded_integrations as $filename => $dependency_class ) {

			$filename = sanitize_file_name( $filename );

			if ( class_exists( $dependency_class ) ) {

				if ( file_exists( WPF_EVENT_TRACKING_DIR_PATH . 'includes/integrations/' . $filename . '/class-' . $filename . '.php' ) ) {
					include WPF_EVENT_TRACKING_DIR_PATH . 'includes/integrations/' . $filename . '/class-' . $filename . '.php';
				}
			}
		}

		// Core integrations.
		include WPF_EVENT_TRACKING_DIR_PATH . 'includes/integrations/core/class-user.php';

		do_action( 'wpf_event_tracking_integrations_loaded' );

	}

	/**
	 * Helper for accessing a single integration.
	 *
	 * @since  1.0.0
	 *
	 * @param  string $integration_name The integration name.
	 * @return class|bool The integration class or false.
	 */
	public function integration( $integration_name ) {

		if ( isset( self::$instance->integrations->{ $integration_name } ) ) {
			return self::$instance->integrations->{ $integration_name };
		} else {
			return false;
		}

	}

	/**
	 * Check install.
	 *
	 * Checks if WP Fusion plugin is active, configured correctly, and if it
	 * supports the user's chosen CRM. If not, returns error message defining
	 * failure.
	 *
	 * @since  1.0.0
	 *
	 * @return mixed True on success, WP_Error on error
	 */
	public function check_install() {
		if ( ! function_exists( 'wp_fusion' ) || ! is_object( wp_fusion()->crm ) || ! wp_fusion()->is_full_version() ) {
			return new WP_Error( 'error', 'WP Fusion is required for <strong>WP Fusion - Event Tracking Addon</strong> to work.' );
		}

		if ( ! wp_fusion()->settings->get( 'connection_configured' ) ) {
			return new WP_Error( 'warning', 'WP Fusion must be connected to a CRM for <strong>WP Fusion - Enhanced Event Tracking Addon</strong> to work. Please deactivate the WP Fusion - Enhanced Event Tracking Addon plugin.' );
		}

		if ( version_compare( WP_FUSION_VERSION, '3.38.23', '<' ) ) {
			return new WP_Error( 'error', 'The Event Tracking Addon requires at least WP Fusion v3.38.23.' );
		}

		if ( ! in_array( 'events', wp_fusion()->crm->supports ) ) {
			return new WP_Error( 'error', 'The WP Fusion Event Tracking addon does not currently support ' . wp_fusion()->crm->name . '. Please deactivate the <strong>WP Fusion - Event Tracking</strong> plugin.' );
		}

		return true;
	}


	/**
	 * Show error message if install check failed.
	 *
	 * @since  1.0.0
	 *
	 * @return mixed error message.
	 */
	public function admin_notices() {

		$return = self::$instance->check_install();

		if ( is_wp_error( $return ) && 'error' === $return->get_error_code() ) {

			echo '<div class="notice notice-error">';
			echo '<p>' . wp_kses_post( $return->get_error_message() ) . '</p>';
			echo '</div>';

		}

	}

	/**
	 * Set up EDD plugin updater.
	 *
	 * @since 1.0.0
	 */
	public function updater() {

		// To support auto-updates, this needs to run during the wp_version_check cron job for privileged users.

		$doing_cron = defined( 'DOING_CRON' ) && DOING_CRON;

		if ( ! current_user_can( 'manage_options' ) && ! $doing_cron ) {
			return;
		}

		$license_status = wp_fusion()->settings->get( 'license_status' );
		$license_key    = wp_fusion()->settings->get( 'license_key' );

		if ( 'valid' === $license_status ) {

			$edd_updater = new WPF_Plugin_Updater(
				WPF_STORE_URL,
				__FILE__,
				array(
					'version' => WPF_EVENT_TRACKING_VERSION,
					'license' => $license_key,
					'item_id' => 83013,
					'author'  => 'Very Good Plugins',
				)
			);

		} else {

			global $pagenow;

			if ( 'plugins.php' === $pagenow ) {
				add_action(
					'after_plugin_row_' . WPF_EVENT_TRACKING_PLUGIN_PATH,
					array(
						wp_fusion(),
						'wpf_update_message',
					),
					10,
					3
				);
			}
		}

	}

}


/**
 * The main function responsible for returning the one true WP Fusion Event
 * Tracking Instance to functions everywhere.
 *
 * Use this function like you would a global variable, except without needing to
 * declare the global.
 *
 * @since  1.0.0
 *
 * @return object The one true WP Fusion Event Tracking
 */
function wp_fusion_event_tracking() {

	return WP_Fusion_Event_Tracking::instance();

}

add_action( 'wp_fusion_init', 'wp_fusion_event_tracking' );
