<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * WooCommerce integration.
 *
 * @since 1.0.0
 */
class WPF_ET_WooCommerce extends WPF_ET_Integrations_Base {

	/**
	 * The slug for WP Fusion's module tracking.
	 *
	 * @since 1.0.0
	 * @var string $slug
	 */

	public $slug = 'woocommerce';

	/**
	 * The plugin name for WP Fusion's module tracking.
	 *
	 * @since 1.0.0
	 * @var string $name
	 */
	public $name = 'WooCommerce';

	/**
	 * Get things started.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		add_action( 'wpf_woocommerce_payment_complete', array( $this, 'new_order' ), 10, 3 );
		add_action( 'wpf_woocommerce_panel', array( $this, 'panel_content' ), 30 );
	}

	/**
	 * Gets the triggers for the integration.
	 *
	 * @access protected
	 *
	 * @since  1.0.0
	 *
	 * @return array The triggers.
	 */
	protected function setup_triggers() {

		$triggers = array(
			'purchased_product' => array(
				'name'         => __( 'Purchased Product', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered each time a single product is purchased.', 'wp-fusion-event-tracking' ),
				'post_types'   => array( 'product' ),
				'has_single'   => true,
				'has_global'   => true,
				'option_types' => array( 'product' ),
			),
			'order_placed'      => array(
				'name'         => __( 'Order Placed', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered each time a WooCommerce order is placed.', 'wp-fusion-event-tracking' ),
				'has_global'   => true,
				'placeholder'  => 'Order',
				'option_types' => array( 'order' ),
			),

		);

		return $triggers;
	}

	/**
	 * Override the add_meta_boxes function in the parent class.
	 *
	 * @since 1.2.0
	 */
	public function add_meta_boxes() {}


	/**
	 * Track event when a new order is placed.
	 *
	 * @since 1.0.0
	 *
	 * @param int    $order_id   The order ID.
	 * @param string $contact_id The contact ID created or updated in the CRM.
	 */
	public function new_order( $order_id, $contact_id ) {

		$order = wc_get_order( $order_id );

		// Get user args.
		$user_id = $order->get_user_id();

		if ( ! empty( $user_id ) ) {
			$args = wp_fusion_event_tracking()->integration( 'user' )->get_user_vars( $user_id );
		} else {
			$args = array(
				'user' => array(
					'first_name'   => $order->get_billing_first_name(),
					'last_name'    => $order->get_billing_last_name(),
					'display_name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
					'user_email'   => $order->get_billing_email(),
				),
			);
		}

		foreach ( $order->get_items() as $item ) {

			$events = $this->get_events( 'purchased_product', $item->get_product_id() );

			if ( ! empty( $events ) ) {

				$product_vars = $this->get_product_vars( $item->get_product_id() );

				$args = array_merge( $args, $product_vars );

				foreach ( $events as $event ) {
					$event = $this->replace_tags( $event, $args );
					$this->track_event( $event, $order->get_billing_email() );
				}
			}
		}

		// Global.
		if ( ! empty( $this->get_global_events( 'order_placed' ) ) ) {

			$order_args = $this->get_order_vars( $order_id );
			$args       = array_merge( $args, $order_args );

			foreach ( $this->get_global_events( 'order_placed' ) as $event ) {

				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $order->get_billing_email() );
			}
		}

	}


	/**
	 * Displays the event tracking fields on the single product settings panel.
	 *
	 * @since 1.0.0
	 *
	 * @param int   $post_id The post ID.
	 */
	public function panel_content( $post_id ) {

		echo '<div class="options_group wpf-event">';

		echo '<p class="form-field"><label><strong>' . esc_html__( 'Event Tracking', 'wp-fusion-event-tracking' ) . '</strong></label></p>';

		echo '<p class="form-field"><label for="wpf-track-events">' . esc_html__( 'Track event when purchased', 'wp-fusion' ) . '</label>';

			$this->render_event_tracking_fields( 'purchased_product', $post_id );

		echo '</p>';

		echo '</div>';

	}

	/**
	 * Gets the order options.
	 *
	 * @since  1.0.0
	 *
	 * @return array The order options.
	 */
	public function get_order_options() {

		return array(
			'name'    => __( 'Order', 'wp-fusion-event-tracking' ),
			'type'    => 'order',
			'options' => array(
				array(
					'meta'        => 'id',
					'preview'     => 33,
					'placeholder' => __( 'The order ID', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'date_created',
					'preview'     => gmdate( 'Y-m-d', strtotime( 'yesterday' ) ),
					'placeholder' => __( 'The date the order was created', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'date_modified',
					'preview'     => gmdate( 'Y-m-d', strtotime( '-3 hours' ) ),
					'placeholder' => __( 'The order\'s last modified date', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'status',
					'preview'     => 'processing',
					'placeholder' => __( 'The order status', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'currency',
					'preview'     => 'USD',
					'placeholder' => __( 'The order currency', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'discount_total',
					'preview'     => '52.00',
					'placeholder' => __( 'The total amount of discount', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'shipping_total',
					'preview'     => '33.00',
					'placeholder' => __( 'The total amount of shipping', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'total',
					'preview'     => '235.00',
					'placeholder' => __( 'The total amount of order', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'billing_first_name',
					'preview'     => 'John',
					'placeholder' => __( 'The Customer billing first Name', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'billing_last_name',
					'preview'     => 'Doe',
					'placeholder' => __( 'The Customer billing last Name', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'billing_email',
					'preview'     => 'john.doe@email.com',
					'placeholder' => __( 'The Customer billing email address', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'billing_address',
					'preview'     => '906 Second Avenue',
					'placeholder' => __( 'The Customer billing address', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'billing_city',
					'preview'     => 'New York',
					'placeholder' => __( 'The Customer billing city', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'billing_country',
					'preview'     => 'United States',
					'placeholder' => __( 'The Customer billing country', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'payment_method',
					'preview'     => 'Cash on Delivery',
					'placeholder' => __( 'The order payment method', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'customer_note',
					'preview'     => 'Note Example',
					'placeholder' => __( 'The Customer order note', 'wp-fusion-event-tracking' ),
				),
			),
		);
	}

	/**
	 * Gets the order variables.
	 *
	 * @since  1.0.0
	 *
	 * @param  int $order_id The order ID.
	 * @return array The order variables.
	 */
	public function get_order_vars( $order_id ) {

		$order = wc_get_order( $order_id );

		if ( empty( $order ) ) {
			return array();
		}

		$data = $order->get_data();

		return array(
			'order' => array(
				'id'                 => $data['id'],
				'date_created'       => $data['date_created']->date_i18n(),
				'date_modified'      => $data['date_modified']->date_i18n(),
				'status'             => $data['status'],
				'currency'           => $data['currency'],
				'discount_total'     => $data['discount_total'],
				'shipping_total'     => $data['shipping_total'],
				'total'              => $data['total'],
				'billing_first_name' => $data['billing']['first_name'],
				'billing_last_name'  => $data['billing']['last_name'],
				'billing_email'      => $data['billing']['email'],
				'billing_address'    => $data['billing']['address_1'],
				'billing_city'       => $data['billing']['city'],
				'billing_country'    => $data['billing']['country'],
				'payment_method'     => $data['payment_method'],
				'customer_note'      => $data['customer_note'],
			),
		);
	}

	/**
	 * Gets the product options.
	 *
	 * @since  1.0.0
	 *
	 * @return array The product options.
	 */
	public function get_product_options() {

		return array(
			'name'    => __( 'Product', 'wp-fusion-event-tracking' ),
			'type'    => 'product',
			'options' => array(
				array(
					'meta'        => 'id',
					'preview'     => 55,
					'placeholder' => __( 'The product ID', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'title',
					'preview'     => 'T-shirt',
					'placeholder' => __( 'The product title', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'date_created',
					'preview'     => gmdate( 'Y-m-d', strtotime( '-1 year' ) ),
					'placeholder' => __( 'The date the product was created', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'date_modified',
					'preview'     => gmdate( 'Y-m-d', strtotime( 'yesterday' ) ),
					'placeholder' => __( 'The product\'s last modified date', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'short_description',
					'preview'     => 'This product is cool.',
					'placeholder' => __( 'The product short description', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'sku',
					'preview'     => 'KA21',
					'placeholder' => __( 'The product SKU', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'price',
					'preview'     => '52.00',
					'placeholder' => __( 'The product price', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'sale_price',
					'preview'     => '33.00',
					'placeholder' => __( 'The product sale price', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'total_sales',
					'preview'     => 64,
					'placeholder' => __( 'The total number of times the product has been sold', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'stock_quantity',
					'preview'     => 55,
					'placeholder' => __( 'The product\'s stock quantity', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'stock_status',
					'preview'     => 'instock',
					'placeholder' => __( 'The product\'s stock status', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'categories',
					'preview'     => 'Clothing, Music',
					'placeholder' => __( 'The product categories (comma separated)', 'wp-fusion-event-tracking' ),
				),
			),
		);
	}

	/**
	 * Gets the product variables.
	 *
	 * @since  1.0.0
	 *
	 * @param  int $product_id The product ID.
	 * @return array The product variables.
	 */
	public function get_product_vars( $product_id ) {

		$product = wc_get_product( $product_id );

		if ( empty( $product ) ) {
			return array();
		}

		$data  = $product->get_data();
		$terms = wp_get_post_terms( $product_id, 'product_cat', array( 'fields' => 'names' ) );
		$vars  = array(
			'product' => array(
				'id'                => $data['id'],
				'title'             => $data['name'],
				'name'              => $data['name'],
				'slug'              => $data['slug'],
				'featured'          => $data['featured'],
				'short_description' => $data['short_description'],
				'sku'               => $data['sku'],
				'price'             => $data['price'],
				'regular_price'     => $data['regular_price'],
				'sale_price'        => $data['sale_price'],
				'total_sales'       => $data['total_sales'],
				'stock_quantity'    => $data['stock_quantity'],
				'stock_status'      => $data['stock_status'],
				'categories'        => implode( ', ', $terms ),
			),
		);

		if ( ! empty( $data['date_created'] ) ) {
			$vars['product']['date_created']  = $data['date_created']->date_i18n();
			$vars['product']['date_modified'] = $data['date_modified']->date_i18n();
		}

		return $vars;

	}


}

new WPF_ET_WooCommerce();
