<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
/**
 * Easy Digital Downloads Recurring Subscription.
 *
 * @since 1.3.0
 */
class WPF_ET_EDD_Recurring extends WPF_ET_Integrations_Base {

	/**
	 * The slug for WP Fusion's module tracking.
	 *
	 * @since 1.3.0
	 * @var string $slug
	 */

	public $slug = 'edd-recurring';

	/**
	 * The plugin name for WP Fusion's module tracking.
	 *
	 * @since 1.3.0
	 * @var string $name
	 */
	public $name = 'EDD Recurring Payments';

	/**
	 * Get things started.
	 *
	 * @since 1.3.0
	 */
	public function init() {

		add_action( 'edd_subscription_status_change', array( $this, 'subscription_change' ), 10, 3 );
		add_action( 'wpf_event_tracking_edd_meta_box', array( $this, 'meta_box_callback' ) );

	}

	/**
	 * Gets the triggers for the integration.
	 *
	 * @access protected
	 *
	 * @since  1.3.0
	 *
	 * @return array The triggers.
	 */
	protected function setup_triggers() {

		$triggers = array(
			'edd_subscription_status_changed' => array(
				'name'         => __( ' Subscription Status Changed', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered each time a subscription status changes.', 'wp-fusion-event-tracking' ),
				'post_types'   => array( 'download' ),
				'has_single'   => true,
				'has_global'   => true,
				'option_types' => array( 'download', 'subscription' ),
			),
		);

		return $triggers;
	}

	/**
	 * Override the add_meta_boxes function in the parent class.
	 *
	 * @since 1.3.0
	 */
	public function add_meta_boxes() {}

	/**
	 * Triggered when a subscription status has changed.
	 *
	 * @since 1.3.0
	 *
	 * @param string $old_status   The old status.
	 * @param string $new_status   The new status.
	 * @param object $subscription The subscription.
	 */
	public function subscription_change( $old_status, $new_status, $subscription ) {

		if ( empty( $subscription->customer ) ) {
			return; // new subs initially have an empty customer.
		}

		$events = $this->get_events( 'edd_subscription_status_changed', $subscription->id );

		if ( ! empty( $events ) ) {

			$args = $this->get_subscription_vars( $subscription->id );

			// Override user vars in case the admin was changing the status of the subscription from the dashboard.
			if ( $subscription->customer->user_id ){
				$user_vars = wp_fusion_event_tracking()->integration( 'user' )->get_user_vars( $subscription->customer->user_id );
				$args      = array_merge( $args, $user_vars );
			}

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $subscription->customer->email );
			}
		}

	}

	/**
	 * Get subscription options
	 *
	 * @since 1.3.0
	 *
	 * @return array
	 */
	public function get_subscription_options() {
		return array(
			'name'    => __( 'Subscription', 'wp-fusion-event-tracking' ),
			'type'    => 'subscription',
			'options' => array(
				array(
					'meta'        => 'id',
					'preview'     => 55,
					'placeholder' => __( 'The subscription ID.', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'period',
					'preview'     => 'month',
					'placeholder' => __( 'The recurring period.', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'initial_amount',
					'preview'     => '9.99',
					'placeholder' => __( 'The initial amount paid.', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'recurring_amount',
					'preview'     => '9.99',
					'placeholder' => __( 'The recurring amount of the subscription.', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'status',
					'preview'     => 'active',
					'placeholder' => __( 'The subscription status.', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'created',
					'preview'     => gmdate( 'Y-m-d', strtotime( '-1 year' ) ),
					'placeholder' => __( 'The subscription created date.', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'expiration',
					'preview'     => gmdate( 'Y-m-d', strtotime( '+1 year' ) ),
					'placeholder' => __( 'The subscription expiration date.', 'wp-fusion-event-tracking' ),
				),
			),
		);

	}

	/**
	 * Gets the details from the subscription.
	 *
	 * @since  1.3.0
	 *
	 * @param  int $subscirption_id The subscription ID.
	 * @return array The subscription variables.
	 */
	public function get_subscription_vars( $subscirption_id ) {

		$subscription = new EDD_Subscription( $subscirption_id );
		return array(
			'subscription' => array(
				'id'               => $subscirption_id,
				'period'           => $subscription->period,
				'initial_amount'   => $subscription->initial_amount,
				'recurring_amount' => $subscription->recurring_amount,
				'created'          => $subscription->created,
				'expiration'       => $subscription->expiration,
				'status'           => $subscription->status,
			),
		);
	}


}

new WPF_ET_EDD_Recurring();
