<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
/**
 * User integration.
 *
 * @since 1.1.0
 */
class WPF_ET_User extends WPF_ET_Integrations_Base {

	/**
	 * The slug for WP Fusion's module tracking.
	 *
	 * @since 1.1.0
	 * @var string $slug
	 */

	public $slug = 'user';

	/**
	 * The plugin name for WP Fusion's module tracking.
	 *
	 * @since 1.1.0
	 * @var string $name
	 */
	public $name = 'User';

	/**
	 * Stores any global options which are shared across integrations.
	 *
	 * @since 1.1.0
	 * @var  array $global_options
	 */
	public $global_option_types = array( 'user' );

	/**
	 * Get things started.
	 *
	 * @since 1.1.0
	 */
	public function init() {

		add_action( 'wp_login', array( $this, 'login' ) );

	}

	/**
	 * Gets the triggers for the integration.
	 *
	 * @access protected
	 *
	 * @since  1.1.0
	 *
	 * @return array The triggers.
	 */
	protected function setup_triggers() {

		$triggers = array(
			'logged_in'  => array(
				'name'         => __( 'Logged In', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered each time a user logs in.', 'wp-fusion-event-tracking' ),
				'has_global'   => true,
				'option_types' => array( 'user' ),
			),
		);

		return $triggers;
	}

	/**
	 * Track an event on login.
	 *
	 * @since 1.1.0
	 *
	 * @param string $user_login The user login.
	 */
	public function login( $user_login ) {

		$user   = get_user_by( 'login', $user_login );
		$events = $this->get_events( 'logged_in', $user->ID );

		if ( ! empty( $events ) ) {

			$args = $this->get_user_vars( $user->ID );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $user->user_email );
			}
		}

	}

	/**
	 * Gets the user options.
	 *
	 * @since  1.1.0
	 *
	 * @return array The download options.
	 */
	public function get_user_options() {

		return array(
			'name'    => __( 'User', 'wp-fusion-event-tracking' ),
			'type'    => 'user',
			'options' => array(
				array(
					'meta'        => 'id',
					'preview'     => 9,
					'placeholder' => __( 'The user ID', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'user_email',
					'preview'     => 'example@email.com',
					'placeholder' => __( 'The user\'s email address', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'display_name',
					'preview'     => 'Jane Doe',
					'placeholder' => __( 'The user\'s display name', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'first_name',
					'preview'     => 'Jane',
					'placeholder' => __( 'The user\'s first name', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'last_name',
					'preview'     => 'Doe',
					'placeholder' => __( 'The user\'s last name', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'user_login',
					'preview'     => 'janeDoe',
					'placeholder' => __( 'The the user\'s username', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'roles',
					'preview'     => 'Editor, Shop Manager',
					'placeholder' => __( 'The user\'s roles', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => '*meta*',
					'placeholder' => __( 'Any user meta key', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'current_url',
					'preview'     => home_url( '/page/' ),
					'placeholder' => __( 'The user\'s current URL', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'referer',
					'preview'     => home_url( '/previous-page/' ),
					'placeholder' => __( 'The user\'s previous URL', 'wp-fusion-event-tracking' ),
				),
			),
		);

	}

	/**
	 * Gets the details from the user for merging.
	 *
	 * At the moment this passes the product ID in the admin when editing a
	 * WooCommerce product. Will have to find a way to fix that later.
	 *
	 * @since  1.1.0
	 *
	 * @param  bool|int $user_id The user ID.
	 * @return array    The user data.
	 */
	public function get_user_vars( $user_id = false ) {

		if ( empty( $user_id ) || empty( get_user_by( 'id', $user_id ) ) ) {

			$user_id = wpf_get_current_user_id();
		}

		$user_meta = wp_fusion()->user->get_user_meta( $user_id );

		foreach ( $user_meta as $key => $value ) {

			if ( 'wp_capabilities' === $key ) {
				$user_meta['role'] = implode( ', ', $value );
			}

			if ( is_array( $value ) ) {
				unset( $user_meta[ $key ] ); // no arrays for now.
			}
		}

		// Page / leadsource stuff.

		if ( ! empty( $_SERVER['REQUEST_URI'] ) ) {
			$user_meta['current_url'] = home_url( $_SERVER['REQUEST_URI'] );
		}

		if ( ! empty( $_SERVER['HTTP_REFERER'] ) ) {
			$user_meta['referer'] = $_SERVER['HTTP_REFERER'];
		}

		$user_meta['id'] = $user_id;

		return array(
			'user' => $user_meta,
		);
	}


}

new WPF_ET_User();
