<?php

class WPF_EC_Hubspot {

	/**
	 * Lets other integrations know which features are supported by the CRM
	 */

	public $supports = array( 'deal_stages', 'products' );

	/**
	 * Get things started
	 *
	 * @access  public
	 * @since   1.0
	 */

	public function init() {

		add_filter( 'wpf_configure_settings', array( $this, 'register_settings' ), 15, 2 );
		add_filter( 'wpf_configure_settings', array( $this, 'hide_attributes_setting' ), 50, 2 );

		add_action( 'wpf_sync', array( $this, 'sync_pipelines' ) );
		add_action( 'wpf_sync', array( $this, 'sync_products' ) );

		add_action( 'admin_init', array( $this, 'maybe_do_initial_sync' ) );

	}

	/**
	 * Do initial sync when the addon is installed or upgraded
	 *
	 * @access public
	 * @since  1.17
	 * @return void
	 */

	public function maybe_do_initial_sync() {

		if ( true == wpf_get_option( 'connection_configured' ) ) {

			$pipelines = wpf_get_option( 'hubspot_pipelines' );

			if ( false === $pipelines ) {
				$this->sync_pipelines();
			}

			$products = get_option( 'wpf_hubspot_products' );

			if ( false === $products ) {
				$this->sync_products();
			}
		}

	}


	/**
	 * Add fields to settings page
	 *
	 * @access public
	 * @return array Settings
	 */

	public function register_settings( $settings, $options ) {

		$settings['ecommerce_header'] = array(
			'title'   => __( 'HubSpot Ecommerce Tracking', 'wp-fusion' ),
			'type'    => 'heading',
			'section' => 'ecommerce',
		);

		if ( ! isset( $options['hubspot_pipelines'] ) ) {
			$options['hubspot_pipelines'] = array();
		}

		$settings['hubspot_pipeline_stage'] = array(
			'title'       => __( 'Pipeline / Stage', 'wp-fusion' ),
			'type'        => 'select',
			'section'     => 'ecommerce',
			'placeholder' => __( 'Select a Pipeline / Stage', 'wp-fusion' ),
			'choices'     => $options['hubspot_pipelines'],
			'std'         => 'default+closedwon',
			'desc'        => __( 'Select a default pipeline and stage for new deals.', 'wp-fusion' ),
		);

		$settings['hubspot_sync_products'] = array(
			'title'   => __( 'Sync Products', 'wp-fusion' ),
			'desc'    => __( 'Sync products purchased as line items to deals in HubSpot.', 'wp-fusion' ),
			'std'     => 1,
			'type'    => 'checkbox',
			'section' => 'ecommerce',
			'tooltip' => __( 'Note that every line item requires a separate API call, so this may not be reliable on stores where orders contain a large number of items (10+).', 'wp-fusion' ),
		);

		$settings['hubspot_add_note'] = array(
			'title'   => __( 'Add Note', 'wp-fusion' ),
			'desc'    => __( 'Add a note to new deals containing the products purchased and prices (Legacy Feature).', 'wp-fusion' ),
			'type'    => 'checkbox',
			'section' => 'ecommerce',
		);

		return $settings;

	}

	/**
	 * Hubspot doesn't support line items distinct from products so we'll hide the setting for now
	 *
	 * @access public
	 * @return array Settings
	 */

	public function hide_attributes_setting( $settings, $options ) {

		if ( isset( $settings['ec_woo_attributes'] ) ) {
			unset( $settings['ec_woo_attributes'] );
			unset( $settings['ec_woo_header'] );
		}

		return $settings;

	}


	/**
	 * Syncs pipelines on plugin install or when Resynchronize is clicked
	 *
	 * @access public
	 * @since  1.0
	 * @return array Pipelines
	 */

	public function sync_pipelines() {

		$pipelines = array();

		$params   = wp_fusion()->crm->get_params();
		$response = wp_remote_get( 'https://api.hubapi.com/crm-pipelines/v1/pipelines/deals/', $params );

		$response = json_decode( wp_remote_retrieve_body( $response ) );

		if ( is_wp_error( $response ) ) {

			wpf_log( 'error', 0, 'Error syncing pipelines: ' . $response->get_error_message(), array( 'source' => 'wpf-ecommerce' ) );

			wp_fusion()->settings->set( 'hubspot_pipelines', $pipelines );

			return $response;
		}

		foreach ( $response->results as $pipeline ) {

			foreach ( $pipeline->stages as $stage ) {

				$pipelines[ $pipeline->pipelineId . '+' . $stage->stageId ] = $pipeline->label . ' &raquo; ' . $stage->label;

			}
		}

		wp_fusion()->settings->set( 'hubspot_pipelines', $pipelines );

		return $pipelines;

	}


	/**
	 * Syncs products on plugin install or when Resynchronize is clicked
	 *
	 * @access public
	 * @since  1.17
	 * @return array Products
	 */

	public function sync_products() {

		$products = array();

		$proceed = true;

		$params  = wp_fusion()->crm->get_params();
		$request = 'https://api.hubapi.com/crm/v3/objects/products?limit=100'; // Must be 100 or you get a 'You can only request at most 100 objects in one request.' error

		while ( $proceed ) {

			$response = wp_remote_get( $request, $params );
			$response = json_decode( wp_remote_retrieve_body( $response ) );

			if ( is_wp_error( $response ) ) {

				wpf_log( 'error', 0, 'Error syncing products: ' . $response->get_error_message(), array( 'source' => 'wpf-ecommerce' ) );

				update_option( 'wpf_hubspot_products', array(), false ); // Update it so it doesn't check again on every page load

				return $response;
			}

			foreach ( $response->results as $product ) {
				$products[ $product->id ] = $product->properties->name;
			}

			if ( count( $response->results ) < 100 ) {
				$proceed = false;
			} else {

				// There are more records
				$request = $response->paging->next->link;
			}
		}

		update_option( 'wpf_hubspot_products', $products, false );

		return $products;

	}

	/**
	 * Register a product in HubSpot
	 *
	 * @access  public
	 * @since   1.17
	 * @return  int Product ID
	 */

	public function add_product( $product ) {

		// See if one exists already with that name
		$products = get_option( 'wpf_hubspot_products', array() );
		$search   = array_search( $product['name'], $products );

		if ( ! empty( $search ) ) {

			update_post_meta( $product['id'], 'hubspot_product_id', $search );
			return $search;

		}

		$product_data = array(
			'properties' => array(
				'name'  => $product['name'],
				'price' => $product['price'],
			),
		);

		if ( ! empty( $product['sku'] ) ) {
			$product_data['properties']['hs_sku'] = $product['sku'];
		}

		wpf_log(
			'info', 0, 'Registering new product <a href="' . admin_url( 'post.php?post=' . $product['id'] . '&action=edit' ) . '" target="_blank">' . $product['name'] . '</a> in HubSpot:', array(
				'meta_array_nofilter' => $product_data,
				'source'              => 'wpf-ecommerce',
			)
		);

		// Add new product
		$params         = wp_fusion()->crm->get_params();
		$params['body'] = wp_json_encode( $product_data );
		$response       = wp_remote_post( 'https://api.hubapi.com/crm/v3/objects/products', $params );

		if ( is_wp_error( $response ) ) {

			if ( false !== strpos( $response->get_error_message(), 'already has that value' ) ) {

				$original_error_message = $response->get_error_message();

				// An existing product already exists.

				$search = array(
					'filterGroups' => array(
						array( // This works out as an OR - https://developers.hubspot.com/docs/api/crm/search.
							'filters' => array(
								array(
									'propertyName' => 'name',
									'operator'     => 'EQ',
									'value'        => str_replace( '"', '', $product['name'] ), // HubSpot throws an error with mismatched quotation marks.
								),
							),
						),
					),
				);

				if ( ! empty( $product['sku'] ) ) {

					$search['filterGroups'][] = array(
						'filters' => array(
							array(
								'propertyName' => 'hs_sku',
								'operator'     => 'EQ',
								'value'        => $product['sku'],
							),
						),
					);

				}

				$params['body'] = wp_json_encode( $search );

				$request  = 'https://api.hubapi.com/crm/v3/objects/products/search';
				$response = wp_remote_post( $request, $params );

				if ( is_wp_error( $response ) ) {
					return $response;
				}

				$response = json_decode( wp_remote_retrieve_body( $response ) );

				if ( ! empty( $response->results ) ) {

					// We found a product with the same SKU

					update_post_meta( $product['id'], 'hubspot_product_id', $response->results[0]->id );

					$products[ $response->results[0]->id ] = $product['name'];
					update_option( 'wpf_hubspot_products', $products, false );

					return $response->results[0]->id;

				} else {

					return new WP_Error( 'error', 'A duplicate record error was triggered while trying to register a new product, but WP Fusion was unable to find a duplicate product via search. This product will not be synced. <stromg>Error message:</strong> ' . $original_error_message );

				}
			} else {

				// Generic error
				return $response;

			}
		}

		$response = json_decode( wp_remote_retrieve_body( $response ) );

		// Save the ID to the product
		update_post_meta( $product['id'], 'hubspot_product_id', $response->id );

		// Update the global products list
		$products[ $response->id ] = $product['name'];
		update_option( 'wpf_hubspot_products', $products, false );

		return $response->id;

	}

	/**
	 * Register a line item in HubSpot.
	 *
	 * @since  1.17
	 * @since  1.17.10 Added $order args.
	 *
	 * @param  array        $product  The product data from the ecommerce
	 *                                integration.
	 * @param  int          $order_id The order ID.
	 * @return int|WP_Error Line Item ID or error.
	 */

	public function add_line_item( $product, $order_id ) {

		$line_item_data = array(
			'properties' => array(
				'hs_product_id' => $product['crm_product_id'],
				'name'          => $product['name'],
				'price'         => $product['price'],
				'quantity'      => $product['qty'],
			),
		);

		/**
		 * Filters the line item data.
		 *
		 * @since 1.17.9
		 * @since 1.17.10 Added $order_id.
		 *
		 * @param array $line_item_data The array of data used to create the line item.
		 * @param array $product        The product data.
		 * @param int   $order_id       The order ID.
		 */

		$line_item_data = apply_filters( 'wpf_ecommerce_hubspot_add_line_item', $line_item_data, $product, $order_id );

		$params         = wp_fusion()->crm->get_params();
		$params['body'] = wp_json_encode( $line_item_data );
		$response       = wp_remote_post( 'https://api.hubapi.com/crm/v3/objects/line_items', $params );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response = json_decode( wp_remote_retrieve_body( $response ) );

		return $response->id;

	}

	/**
	 * Add an order.
	 *
	 * @access  public
	 * @return  bool
	 */

	public function add_order( $order_id, $contact_id, $order_args ) {

		if ( empty( $order_args['order_date'] ) ) {
			$order_date = current_time( 'timestamp' );
		} else {
			$order_date = $order_args['order_date'];
		}

		$calc_totals = 0;

		// Build up items array.
		foreach ( $order_args['products'] as $product ) {

			if ( ! isset( $product['price'] ) ) {
				$product['price'] = 0;
			}

			$calc_totals += $product['qty'] * $product['price'];

		}

		// Adjust total for line items.
		foreach ( $order_args['line_items'] as $line_item ) {
			$calc_totals += $line_item['price'];
		}

		$pipeline_stage = wpf_get_option( 'hubspot_pipeline_stage', 'default+closedwon' );

		// See if there's a custom pipeline stage set for Woo.

		if ( isset( $order_args['status'] ) && ! empty( wpf_get_option( "ec_woo_status_wc-{$order_args['status']}" ) ) ) {
			$pipeline_stage = wpf_get_option( "ec_woo_status_wc-{$order_args['status']}" );
		}

		$pipeline_stage = explode( '+', $pipeline_stage );

		$order = array(
			'associations' => array(
				'associatedVids' => array( $contact_id ),
			),
			'properties'   => array(
				array(
					'name'  => 'dealname',
					'value' => $order_args['order_label'],
				),
				array(
					'name'  => 'pipeline',
					'value' => $pipeline_stage[0],
				),
				array(
					'name'  => 'dealstage',
					'value' => $pipeline_stage[1],
				),
				array(
					'name'  => 'closedate',
					'value' => $order_date * 1000,
				),
				array(
					'name'  => 'amount',
					'value' => round( $calc_totals, 2 ),
				),
			),
		);

		/**
		 * Filters the deal data.
		 *
		 * @since 1.15.0
		 *
		 * @link https://wpfusion.com/documentation/ecommerce-tracking/hubspot-ecommerce/#custom-deal-fields
		 *
		 * @param array $order    The deal.
		 * @param int   $order_id ID of the order.
		 */

		$order = apply_filters( 'wpf_ecommerce_hubspot_add_deal', $order, $order_id );

		if ( empty( $order ) ) {
			return false; // allow for skipping.
		}

		$existing_deal_id = get_post_meta( $order_id, 'wpf_ec_hubspot_invoice_id', true );

		if ( empty( $existing_deal_id ) ) {
			$message = 'Creating deal for <a href="' . $order_args['order_edit_link'] . '" target="_blank">' . $order_args['order_label'] . '</a>:';
		} else {
			$message = 'Updating HubSpot deal #' . $existing_deal_id . ' from <a href="' . $order_args['order_edit_link'] . '" target="_blank">' . $order_args['order_label'] . '</a>:';
		}

		wpf_log(
			'info',
			$order_args['user_id'],
			$message,
			array(
				'meta_array_nofilter' => $order,
				'source'              => 'wpf-ecommerce',
			)
		);

		$params         = wp_fusion()->crm->get_params();
		$params['body'] = wp_json_encode( $order );

		if ( empty( $existing_deal_id ) ) {
			$response = wp_remote_post( 'https://api.hubapi.com/deals/v1/deal/', $params );
		} else {
			$params['method'] = 'PUT';
			$response         = wp_remote_request( 'https://api.hubapi.com/deals/v1/deal/' . $existing_deal_id, $params );
		}

		if ( is_wp_error( $response ) ) {

			if ( false !== strpos( $response->get_error_message(), 'No deal found for' ) ) {

				// We tried to update a deal that's been deleted. Clear the saved ID and start over.
				delete_post_meta( $order_id, 'wpf_ec_hubspot_invoice_id' );
				return $this->add_order( $order_id, $contact_id, $order_args );

			}

			wpf_log( $response->get_error_code(), $order_args['user_id'], 'Error adding order: ' . $response->get_error_message(), array( 'source' => wp_fusion()->crm->slug ) );
			return $response;

		}

		$response = json_decode( wp_remote_retrieve_body( $response ) );

		if ( ! empty( $existing_deal_id ) ) {
			return; // If we've just updated an existing deal we don't need to attach a note or modify the line items.
		}

		$deal_id = $response->dealId;

		if ( wpf_get_option( 'hubspot_add_note' ) ) {

			// Attach note to the deal
			$body = '';

			foreach ( $order_args['products'] as $product ) {

				$body .= $product['name'] . ' - ' . $order_args['currency_symbol'] . $product['price'];

				if ( $product['qty'] > 1 ) {
					$body .= ' - x' . $product['qty'];
				}

				$body .= '<br/>';

			}

			foreach ( $order_args['line_items'] as $line_item ) {

				$body .= $line_item['title'] . ' - ' . $order_args['currency_symbol'] . $line_item['price'] . '<br />';

			}

			$engagement_data = array(
				'engagement'   => array(
					'type' => 'NOTE',
				),
				'associations' => array(
					'dealIds' => array( $deal_id ),
				),
				'metadata'     => array(
					'body' => $body,
				),
			);

			/**
			 * Filters the engagement data.
			 *
			 * @since 1.17.9
			 *
			 * @param array $engagement_data The array of data used to create the engagement.
			 * @param int   $order_id        The order ID.
			 */

			$engagement_data = apply_filters( 'wpf_ecommerce_hubspot_add_engagement', $engagement_data, $order_id );

			wpf_log(
				'info', $order_args['user_id'], 'Adding engagement data (deal notes) to order <a href="' . $order_args['order_edit_link'] . '" target="_blank">#' . $order_id . '</a>:', array(
					'meta_array_nofilter' => $engagement_data,
					'source'              => 'wpf-ecommerce',
				)
			);

			$params['body'] = wp_json_encode( $engagement_data );

			$response = wp_remote_post( 'https://api.hubapi.com/engagements/v1/engagements', $params );

			if ( is_wp_error( $response ) ) {

				wpf_log( 'error', $order_args['user_id'], 'Error adding note to deal: ' . $response->get_error_message(), array( 'source' => 'wpf-ecommerce' ) );
				return $deal_id;

			}
		}

		if ( wpf_get_option( 'hubspot_sync_products' ) ) {

			// Sync products and line items
			foreach ( $order_args['products'] as $product ) {

				if ( empty( $product['crm_product_id'] ) ) {

					// Get the product ID
					$response = $this->add_product( $product );

					if ( is_wp_error( $response ) ) {

						wpf_log(
							'error', $order_args['user_id'], 'Error registering new product in HubSpot: ' . $response->get_error_message(), array(
								'source'              => 'wpf-ecommerce',
								'meta_array_nofilter' => $product,
							)
						);
						continue;

					} else {

						$product['crm_product_id'] = $response;

					}
				}

				// Add the line item
				$line_item_id = $this->add_line_item( $product, $order_id );

				if ( is_wp_error( $line_item_id ) ) {

					wpf_log(
						'error', $order_args['user_id'], 'Error creating line item: ' . $line_item_id->get_error_message(), array(
							'source'              => 'wpf-ecommerce',
							'meta_array_nofilter' => $product,
						)
					);
					continue;
				}

				$params           = wp_fusion()->crm->get_params();
				$params['method'] = 'PUT';

				// Associate the line item with the deal
				$response = wp_remote_request( 'https://api.hubapi.com/crm/v3/objects/line_items/' . $line_item_id . '/associations/DEAL/' . $deal_id . '/LINE_ITEM_TO_DEAL', $params );

				if ( is_wp_error( $response ) ) {
					wpf_log( 'error', $order_args['user_id'], 'Error associating line item ID ' . $line_item_id . ' with deal ID ' . $deal_id . ': ' . $response->get_error_message(), array( 'source' => 'wpf-ecommerce' ) );
					continue;
				}

				$response = json_decode( wp_remote_retrieve_body( $response ) );

				// Done
			}
		}

		return $deal_id;

	}

	/**
	 * Update a deal stage when an order status is changed
	 *
	 * @access  public
	 * @return  bool
	 */

	public function change_stage( $deal_id, $stage, $order_id ) {

		$pipeline_stage = explode( '+', $stage );

		$deal = array(
			'properties' => array(
				array(
					'name'  => 'pipeline',
					'value' => $pipeline_stage[0],
				),
				array(
					'name'  => 'dealstage',
					'value' => $pipeline_stage[1],
				),
			),
		);

		$params = wp_fusion()->crm->get_params();

		/**
		 * Filters the deal when changing the stage.
		 *
		 * @since 1.17.9
		 *
		 * @param array $deal     The deal data.
		 * @param int   $deal_id  The deal ID to be updated.
		 * @param int   $order_id The order ID.
		 */

		$deal = apply_filters( 'wpf_ecommerce_hubspot_change_deal_stage', $deal, $deal_id, $order_id );

		$params['body']   = wp_json_encode( $deal );
		$params['method'] = 'PUT';

		$response = wp_remote_request( 'https://api.hubapi.com/deals/v1/deal/' . $deal_id, $params );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

	}



}
