<?php

// deny direct access
if ( ! function_exists( 'add_action' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit();
}

/**
 * Class WPF_Downloads_Public
 *
 *
 */
class WPF_Downloads_Public {


	private $wpdb;

	/**
	 * Get things started
	 *
	 * @since 1.0
	 * @return void
	 */

	public function __construct() {

		global $wpdb;

		$this->wpdb = $wpdb;

		// setting up the endpoint at init action
		add_action( 'init', array( $this, 'file_endpoint' ) );

		/**
		 * parse_query
		 *
		 * parse_query is an action triggered after WP_Query->parse_query() has set up query variables
		 * WP_Query->parse_query():  Parse a query string and set query type booleans.
		 */
		add_action( 'parse_query', array( $this, 'parse_query' ) );
	}

	/**
	 *  file_endpoint
	 *
	 * we need to add the endpoint so we can control who can access the media files
	 */

	public function file_endpoint() {

		// setting the file_endpoint from WPF_DOWNLOADS_ENDPOINT value
		add_rewrite_endpoint( WPF_DOWNLOADS_ENDPOINT, EP_ROOT );
	}


	/**
	 * parse_query
	 *
	 * parse the query form the WPF_DOWNLOADS_ENDPOINT
	 *
	 * @since 1.0
	 *
	 * @param  $query
	 */

	public function parse_query( $query ) {

		// find WPF_DOWNLOADS_ENDPOINT in $query->query_vars
		if ( isset( $query->query_vars[ WPF_DOWNLOADS_ENDPOINT ] ) ) {

			// check file protection
			$this->check_file_protection();
			exit;
		}
	}

	/**
	 * file_not_found
	 *
	 * returns 404 for protected file or if file doesn't exists
	 */

	public function file_not_found() {

		global $wp_query;

		// Set the 404 property and saves whether query is feed
		$wp_query->set_404();

		$redirect = wp_fusion()->settings->get( 'downloads_default_redirect' );

		if ( ! empty( $redirect ) ) {

			wp_redirect( $redirect );
			exit();

		} else {

			// set HTTP status header to 404.
			status_header( 404 );
			get_template_part( 404 );

		}
	}


	/**
	 * check_file_protection
	 *
	 *
	 */

	public function check_file_protection() {

		// get full file name
		$full_file_name = filter_input( INPUT_GET, WPF_DOWNLOADS_ENDPOINT );

		// get a $file_type from WPF_DOWNLOADS_ENDPOINT INPUT_GET variable
		$pathinfo = pathinfo($full_file_name);

		if ( isset( $pathinfo['extension'] ) ) {
			$file_type = $pathinfo['extension'];
		} else {
			$file_type = filter_input( INPUT_GET, 'file_type' );
		}

		// get a $file_name from WPF_DOWNLOADS_ENDPOINT INPUT_GET variable
		$file_name = str_replace( '.' . $file_type, '', $full_file_name );

		// if $file_name is empty return 404
		if ( empty( $file_name ) ) {

			// if $file_name is empty return 404
			$this->file_not_found();
			return;

		}

		$full_file_name = $file_name . '.' . $file_type;

		$guid = str_replace( '/_' . WPF_DOWNLOADS_ENDPOINT, '', $this->extract_value( $full_file_name, $file_type ) );

		// get post by guid
		$post = $this->get_post_by_guid( $guid );

		// if we could find the post by guid
		if ( isset( $post ) ) {
			// output file if user has the right role
			$this->check_protected_file( $post->ID, $full_file_name );

		} // if we could not find the post by guid
		else {

			// concatenate $file_name and file type, delete the '/' from start of the string
			$meta_value = ltrim( substr( $file_name, 0 ) . '.' . $file_type, '/' );

			// extract the value from $meta_value
			$meta_value = $this->extract_value( $meta_value, $file_type );

			//get post meta my value $meta_value Example: 2019/08/image-1-2.png
			$post_meta = $this->get_post_meta_by_value( $meta_value );

			// if we could get the post meta
			if ( isset( $post_meta ) ) {

				// output file if user has the right role
				$this->check_protected_file( $post_meta->post_id, $full_file_name );

			} // we could not get the post meta
			else {

				// get full file path
				$file_path = $this->get_full_file_path( $full_file_name );

				// output file
				$this->output_file( $file_path );

			}
		}
	}


	/**
	 * extract_value
	 *
	 * get guid from $file_name and $file_type
	 *
	 * @param $guid
	 * @param $file_type
	 *
	 * @return string guid
	 */
	public function extract_value( $guid, $file_type ) {

		$pattern = "/-\d+x\d+.$file_type$/";
		$result  = preg_replace( $pattern, '.' . $file_type, $guid );

		// return guid
		return $result;
	}


	/**
	 * get_post_by_guid
	 *
	 * query the DB to get the post by guid
	 *
	 * @param $guid
	 *
	 * @return object $post
	 */
	public function get_post_by_guid( $guid ) {

		global $wpdb;
		$value = '%' . $guid;

		// get the post by guid
		$post = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM  $wpdb->posts WHERE post_type='attachment' AND guid LIKE %s", $value ) );

		//return $post
		return $post;
	}

	/**
	 * get_post_meta_by_value
	 *
	 * @param $value
	 *
	 * @return object $post
	 */

	public function get_post_meta_by_value( $value ) {

		global $wpdb;
		$post = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $wpdb->postmeta WHERE meta_key='_wp_attached_file' AND meta_value LIKE %s", $value ) );

		//return $post
		return $post;
	}

	/**
	 * check_protected_file
	 *
	 * @param $id
	 * @param $original_file
	 */

	public function check_protected_file( $id, $original_file ) {

		// get the full path if it's allowed role are admin request
		if ( $this->is_admin_request() || wp_fusion()->access->user_can_access( $id ) ) {

			// get full path
			$file_path = $this->get_full_file_path( str_replace( '/_' . WPF_DOWNLOADS_ENDPOINT, '', $original_file ) );

			// output file
			$this->output_file( $file_path );
		} else {

			// if not allowed do 404
			$this->file_not_found();
			return;

		}
	}


	/**
	 * get_full_file_path
	 *
	 * @param $original_file
	 *
	 * @return string
	 */

	public function get_full_file_path( $original_file ) {

		// get the upload dir array
		$wp_upload_dir = wp_upload_dir();

		// get the basedir from upload dir array
		$upload_base_dir = $wp_upload_dir['basedir'];

		// create the file path
		$file_path = $upload_base_dir . '/_' . WPF_DOWNLOADS_ENDPOINT . $original_file;

		//return $file_path
		return $file_path;
	}


	/**
	 * output_file
	 *
	 * @param $file
	 */

	public function output_file( $file ) {

		global $wp_filesystem;

		//check if $wp_filesystem is not initialized yet then initialize it
		if ( empty( $wp_filesystem ) || is_null( $wp_filesystem ) ) {

			require_once ABSPATH . '/wp-admin/includes/file.php';

			WP_Filesystem();
		}

		// check if $file is file
		if ( ! is_file( $file ) ) {
			$this->file_not_found();
			return;
		}

		// get the mime type form the file
		$mimetype = $this->get_mimetype( $file );

		if ( $this->is_unknown_filetype( $file, $mimetype ) ) {
			$file_name = basename( $file );
			header( 'Content-Disposition: attachment; filename=' . $file_name );
		}

		//set header Content-Type by $mimetype
		header( 'Content-Type: ' . $mimetype ); // always send this
		if ( false === strpos( $_SERVER['SERVER_SOFTWARE'], 'Microsoft-IIS' ) ) {
			header( 'Content-Length: ' . filesize( $file ) );
		}

		// set $last_modified
		$last_modified = gmdate( 'D, d M Y H:i:s', filemtime( $file ) );

		/** set ETag. ETags are used for Web cache validation and conditional requests from browsers for resources.
		 * More info available here: https://whatis.techtarget.com/definition/entity-tag-Etag
		 */
		$etag = '"' . md5( $last_modified ) . '"';

		// set header for  $last_modified
		header( "Last-Modified: $last_modified GMT" );

		// set header for ETag
		header( 'ETag: ' . $etag );

		// set header for Expires
		header( 'Expires: ' . gmdate( 'D, d M Y H:i:s', time() + 100000000 ) . ' GMT' );

		/*
		// Not sure if we need this
		//

		// Support for Conditional GET
		$client_etag = isset( $_SERVER['HTTP_IF_NONE_MATCH'] ) ? stripslashes( $_SERVER['HTTP_IF_NONE_MATCH'] ) : false;

		// check HTTP_IF_MODIFIED_SINCE and set accordingly
		if ( ! isset( $_SERVER['HTTP_IF_MODIFIED_SINCE'] ) ) {
			$_SERVER['HTTP_IF_MODIFIED_SINCE'] = false;
		}

		// set $client_last_modified from HTTP_IF_MODIFIED_SINCE
		$client_last_modified = trim( $_SERVER['HTTP_IF_MODIFIED_SINCE'] );

		// If string is empty, return 0. If not, attempt to parse into a timestamp
		$client_modified_timestamp = $client_last_modified ? strtotime( $client_last_modified ) : 0;

		// Make a timestamp for our most recent modification...
		$modified_timestamp = strtotime( $last_modified );

		if ( ( $client_last_modified && $client_etag )
			? ( ( $client_modified_timestamp >= $modified_timestamp ) && ( (string) $client_etag === $etag ) )
			: ( ( $client_modified_timestamp >= $modified_timestamp ) || ( (string) $client_etag === $etag ) )
		) {
			// 304: NOT MODIFIED
			status_header( 304 );
			exit;
		}

		*/

		// 200: request has succeeded
		status_header( 200 );

		/* read the file */
		$read_file = '';
		if ( $wp_filesystem->exists( $file ) ) { //check for existence
			$read_file = $wp_filesystem->get_contents( $file );

			if ( ! $read_file ) {
				return new WP_Error( 'reading_error', 'Error when reading file' );
			} //return error object
		}

		// 200: request has succeeded
		status_header( 200 );
		echo( $read_file );
	}


	/**
	 * get_mimetype
	 *
	 * @param $file
	 *
	 * @return string
	 */

	public function get_mimetype( $file ) {

		// Retrieve the file type from the file name
		$mime = wp_check_filetype( $file );

		// if content type has not ben detected with wp_check_filetype use mime_content_type
		if ( empty( $mime['type'] ) && function_exists( 'mime_content_type' ) ) {
			$mime['type'] = mime_content_type( $file );
		}

		// if $mime['type'] is set
		if ( $mime['type'] ) {

			// set the $mimetype so we can return it
			$mimetype = $mime['type'];

		} else {

			// otherwise set it from the substr of file
			$mimetype = 'image/' . substr( $file, strrpos( $file, '.' ) + 1 );
		}

		// return the $mimetype
		return $mimetype;
	}


	/**
	 * is_unknown_filetype
	 *
	 * @param $file
	 * @param $mimetype
	 *
	 * @return bool
	 */

	public function is_unknown_filetype( $file, $mimetype ) {

		if ( 'application/pdf' === $mimetype ) {
			return false;
		} elseif ( strstr( $mimetype, 'video/' ) ) {
			// video
			return false;
		} elseif ( strstr( $mimetype, 'audio/' ) ) {
			// audio
			return false;
		} elseif ( exif_imagetype( $file ) ) {
			// image
			return false;
		} elseif ( strstr( $mimetype, 'image/' ) ) {
			// image
			return false;
		} else {
			// not recognized
			return true;
		}

	}

	/**
	 * is_admin_request
	 *
	 * check if request is in WP admin
	 *
	 * @return bool
	 */

	public function is_admin_request() {

		// get the currenturl
		$current_url = home_url( add_query_arg( null, null ) );

		// get admin url
		$admin_url = strtolower( admin_url() );

		// get the referer
		$referrer = strtolower( wp_get_referer() );

		/**
		 * Check if this is a admin request. If true, it
		 * could also be a AJAX request from the frontend.
		 */
		if ( strstr( $referrer, 'wp-admin' ) ) {

			return true;
		}
		if ( 0 === strpos( $current_url, $admin_url ) ) {

			// Check if the user comes from a admin page
			if ( 0 === strpos( $referrer, $admin_url ) ) {

				return true;
			} else {
				// Check for AJAX requests.
				if ( function_exists( 'wp_doing_ajax' ) ) {
					return ! wp_doing_ajax();
				} else {
					return ! ( defined( 'DOING_AJAX' ) && DOING_AJAX );
				}
			}
		} else {
			return false;
		}
	}

}

$wpfd_public = new WPF_Downloads_Public();
