<?php

// deny direct access
if ( ! function_exists( 'add_action' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit();
}

if ( ! class_exists( "WPF_Downloads_File" ) ) {
	/**
	 * Class WPF_Downloads_File
	 *
	 * handling files (protecting, un-protecting)
	 */
	class WPF_Downloads_File {

		/**
		 * protect_file
		 *
		 * Protecting file
		 *
		 * @param $post_id
		 *
		 * @return array|bool|int|mixed|WP_Error
		 */
		public function protect_file( $post_id ) {
			// get the file path from the post
			$file = get_post_meta( $post_id, '_wp_attached_file', true );

			// check if file is already protected
			if ( 0 === stripos( $file, $this->mv_upload_dir( '/' ) ) ) {

				return new WP_Error( 'file_already_protected', sprintf(
					__( 'This file is already protected. Please reload your page.', 'wp-fusion' ),
					$file
				), array( 'status' => 500 ) );

			}

			// dir where file is stored for example: 2019/07
			$reldir = dirname( $file );

			// Checks if a value exists in an array
			if ( in_array( $reldir, array( '\\', '/', '.' ), true ) ) {
				$reldir = '';
			}

			// join the _wpfd with reldir value for example: _wpfd/2019/07
			$protected_dir = path_join( $this->mv_upload_dir(), $reldir );

			// move the file to protected dir and get the result
			$move_result = $this->move_attachment_to_protected( $post_id, $protected_dir );

			// if result is error return it, otherwise store the value $is_protected in meta with update_post_meta
			if ( is_wp_error( $move_result ) ) {
				return $move_result;
			} else {
				return $this->updated_file_protection( $post_id, true );
			}
		}

		/**
		 * un_protect_file
		 *
		 * @param $post_id
		 *
		 * @return array|bool|mixed
		 */
		public function un_protect_file( $post_id ) {

			// get the file path from the post
			$file = get_post_meta( $post_id, '_wp_attached_file', true );

			// check if file is already protected
			if ( 0 !== stripos( $file, $this->mv_upload_dir( '/' ) ) ) {
				return true;
			}

			// for example 2019/07
			$protected_dir = ltrim( dirname( $file ), $this->mv_upload_dir( '/' ) );

			// un-protects the file and returns the result
			$move_result = $this->move_attachment_to_protected( $post_id, $protected_dir );


			if ( is_wp_error( $move_result ) ) {
				return $move_result;
			}
		}


		/**
		 * mv_upload_dir
		 *
		 * returns the protected dir _wpfd
		 *
		 * @param string $path
		 * @param bool   $in_url
		 *
		 * @return string
		 */
		public function mv_upload_dir( $path = '', $in_url = false ) {

			$dirpath  = $in_url ? '/' : '';
			$dirpath .= '_wpfd';
			$dirpath .= $path;

			return $dirpath;

		}

		/**
		 * move_attachment_to_protected
		 *
		 * moves the file to protected directory
		 *
		 * @param       $attachment_id
		 * @param       $protected_dir
		 * @param array $meta_input
		 *
		 * @return array|bool|mixed
		 */
		public function move_attachment_to_protected( $attachment_id, $protected_dir, $meta_input = [] ) {

			// check if post type is of type attachment
			if ( 'attachment' !== get_post_type( $attachment_id ) ) {

				return new WP_Error( 'not_attachment', /* translators: post type is not of type attachment */
					sprintf( __( 'The post with ID: %d is not an attachment post type.', 'wp-fusion-downloads' ), $attachment_id ), array( 'status' => 404 ) );
			}

			if ( path_is_absolute( $protected_dir ) ) {
				return new WP_Error( 'protected_dir_not_relative', /* translators: path must be a path relative to the WP uploads directory */
					sprintf( __( 'The new path provided: %s is absolute. The new path must be a path relative to the WP uploads directory.', 'wp-fusion-downloads' ), $protected_dir ), array( 'status' => 404 ) );
			}

			// check $meta_input, get attachment meta
			$meta = empty( $meta_input ) ? wp_get_attachment_metadata( $attachment_id ) : $meta_input;

			// check if $meta is aray
			$meta = is_array( $meta ) ? $meta : array();

			// get file path from post meta
			$file = get_post_meta( $attachment_id, '_wp_attached_file', true );

			// get backup sizes form the attachment
			$backups = get_post_meta( $attachment_id, '_wp_attachment_backup_sizes', true );


			/**
			 * wp_upload_dir()
			 *
			 * get upload dir array that has the indexes:
			 * [path] => /app/public/wp-content/uploads/2019/08
			 * [url] => http://wpfdownloads2.local/wp-content/uploads/2019/08
			 * [subdir] => /2019/08
			 * [basedir] => /app/public/wp-content/uploads
			 * [baseurl] => http://wpfdownloads2.local/wp-content/uploads
			 * [error] =>
			 */
			$upload_dir = wp_upload_dir();

			// directory where the file is
			$old_dir = dirname( $file );

			if ( in_array( $old_dir, array( '\\', '/', '.' ), true ) ) {
				$old_dir = '';
			}

			if ( $protected_dir === $old_dir ) {
				return true;
			}

			// path of the unprotected directory
			$old_full_path = path_join( $upload_dir['basedir'], $old_dir );

			// path of the protected directory
			$protected_full_path = path_join( $upload_dir['basedir'], $protected_dir );

			// wp_mkdir_p does a recursive directory creation based on full path
			if ( ! wp_mkdir_p( $protected_full_path ) ) {
				return new WP_Error( 'wp_mkdir_p_error', sprintf( /* translators: There was an error making or verifying the directory  */
					__( 'There was an error making or verifying the directory at: %s', 'wp-fusion-downloads' ), $protected_full_path ), array( 'status' => 500 ) );
			}


			// check $meta if it has the 'sizes' index
			// Get all files from meta
			$sizes = array();
			if ( array_key_exists( 'sizes', $meta ) ) {

				// Get all files from meta
				$sizes = $this->get_files_from_meta( $meta['sizes'] );
			}
			$backup_sizes = $this->get_files_from_meta( $backups );

			// making $old_basenames and $new_basenames vars for further processing
			/**
			 * $old_basenames
			 * [0] => image-1-2.png
			 * [1] => image-1-2-150x150.png
			 * [2] => image-1-2-300x283.png
			 *
			 * $new_basenames
			 * [0] => image-1-2.png
			 * [1] => image-1-2-150x150.png
			 * [2] => image-1-2-300x283.png
			 */

			$old_basenames = $new_basenames = array_merge( array( basename( $file ) ), $sizes, $backup_sizes );

			// basename of the file for example: image-1.png
			$orig_basename = basename( $file );

			// if thera are values in $backups set $orig_basename of the file for example: image-1.png
			if ( is_array( $backups ) && isset( $backups['full-orig'] ) ) {
				$orig_basename = $backups['full-orig']['file'];
			}

			/**
			 * pathinfo()
			 *
			 * Returns information about a file path
			 *
			 * For example:
			 * [dirname] => .
			 * [basename] => image-1-2.png
			 * [extension] => png
			 * [filename] => image-1-2
			 */

			$orig_filename = pathinfo( $orig_basename );

			// get the filename from $orig_filename
			$orig_filename = $orig_filename['filename'];

			// resolve the name conflict resolve_name_conflict()
			$result        = $this->resolve_name_conflict( $new_basenames, $protected_full_path, $orig_filename );
			$new_basenames = $result['new_basenames'];

			//rename files
			$this->rename_files( $old_basenames, $new_basenames, $old_full_path, $protected_full_path );

			$base_file_name = 0;

			// join $protected_dir $new_basenames[0] paths. For example: 2019/08/image-1-2.png
			$new_attached_file = path_join( $protected_dir, $new_basenames[0] );

			// meta key file exists update it with $new_attached_file protected path
			if ( array_key_exists( 'file', $meta ) ) {
				$meta['file'] = $new_attached_file;
			}

			// update the post meta _wp_attached_file value
			update_post_meta( $attachment_id, '_wp_attached_file', $new_attached_file );


			// if $new_basenames[ $base_file_name ] is not same as $old_basenames[ $base_file_name ]
			// Example: $new_basenames[ $base_file_name ] =  image-1-2.png
			if ( $new_basenames[ $base_file_name ] !== $old_basenames[ $base_file_name ] ) {

				// result from resolve_name_conflict() call line 232
				$pattern   = $result['pattern'];
				$replace   = $result['replace'];
				$separator = '#';

				$orig_basename = ltrim(
					str_replace( $pattern, $replace, $separator . $orig_basename ),
					$separator
				);

				// update the sizes files in $meta with $new_basenames
				$meta = $this->update_meta_sizes_file( $meta, $new_basenames );

				// update the backups files in $meta with $new_basenames
				$this->update_backup_files( $attachment_id, $backups, $new_basenames );
			}

			// updates the _wp_attachment_metadata with updated $meta values
			update_post_meta( $attachment_id, '_wp_attachment_metadata', $meta );

			// Sets the $guid value
			// Example: /app/public/wp-content/uploads/_wpfd/2019/08/image-1-2.png
			$guid = path_join( $protected_full_path, $orig_basename );

			// updates the posts guid with $guid
			$result = wp_update_post( array(
				'ID'   => $attachment_id,
				'guid' => $guid,
			) );

			// check $meta_input return $meta
			return empty( $meta_input ) ? true : $meta;
		}

		/**
		 * updated_file_protection
		 *
		 * Updates post meta with $is_protected value
		 *
		 * @param $post_id
		 * @param $is_protected
		 *
		 * @return bool|int
		 */
		public function updated_file_protection( $post_id, $is_protected ) {

			return update_post_meta( $post_id, 'wpfprotected', $is_protected );
		}

		/**
		 * get_files_from_meta
		 *
		 * Obtains the files (all sizes) from the attachment meta
		 *
		 * @param $input
		 *
		 * @return array
		 */
		public function get_files_from_meta( $input ) {
			$files = array();

			if ( is_array( $input ) ) {

				//loop trough the size files
				foreach ( $input as $size ) {
					$files[] = $size['file'];
				}
			}

			return $files;
		}

		/**
		 * resolve_name_conflict
		 *
		 * resolves name conflict
		 *
		 * @param $new_basenames
		 * for exaple [0] => image-1-2.png
		 *
		 * @param $protected_full_path
		 * path of the protected directory
		 *
		 * @param $orig_file_name
		 * $orig_filename = pathinfo( $orig_basename );
		 * get the filename from $orig_filename
		 * $orig_filename = $orig_filename['filename'];
		 *
		 * @return array
		 */

		public function resolve_name_conflict( $new_basenames, $protected_full_path, $orig_file_name ) {
			$conflict  = true;
			$number    = 1;
			$separator = '#';

			$med_filename = $orig_file_name;
			$pattern      = '';
			$replace      = '';

			while ( $conflict ) {
				$conflict = false;

				// lopping trough all items in $new_basenames
				foreach ( $new_basenames as $basename ) {

					//path_join
					if ( is_file( path_join( $protected_full_path, $basename ) ) ) {
						$conflict = true;
						break;
					}
				}

				// if there is a conflict
				if ( $conflict ) {

					//solve the conflict
					$new_filename = $orig_file_name . '-' . $number;
					$number ++;
					$pattern = $separator . $med_filename;
					$replace = $separator . $new_filename;

					$new_basenames = explode(
						$separator,
						ltrim(
							str_replace( $pattern, $replace, $separator . implode( $separator, $new_basenames ) ),
							$separator
						)
					);

				}
			}

			return array(
				'new_basenames' => $new_basenames,
				'pattern'       => $pattern,
				'replace'       => $replace,
			);
		}

		/**
		 * rename_files
		 *
		 * Loop trough $old_basenames and $new_basenames and rename the files
		 *
		 * @param $old_basenames
		 * @param $new_basenames
		 * @param $old_dir
		 * @param $protected_dir
		 *
		 * @return WP_Error
		 */

		public function rename_files( $old_basenames, $new_basenames, $old_dir, $protected_dir ) {

			$unique_old_basenames = array_values( array_unique( $old_basenames ) );
			$unique_new_basenames = array_values( array_unique( $new_basenames ) );

			$i = count( $unique_old_basenames );

			while ( $i -- ) {

				// join paths
				$old_fullpath = path_join( $old_dir, $unique_old_basenames[ $i ] );
				$new_fullpath = path_join( $protected_dir, $unique_new_basenames[ $i ] );

				if ( is_file( $old_fullpath ) ) {

					// rename
					rename( $old_fullpath, $new_fullpath );

					if ( ! is_file( $new_fullpath ) ) {

						return new WP_Error(

							'rename_failed',

							sprintf(
							/* translators: post type is not of type attachment */
								__( 'Rename failed when trying to move file from: %$1s, to: %$2s', 'wp-fusion' ),
								$old_fullpath,
								$new_fullpath
							)
						);
					}
				}
			}
		}


		/**
		 * update_meta_sizes_file
		 *
		 * Each attachment has a number of thumbnails that are stored in $meta as sizes
		 * This function updates those values
		 *
		 * @param $meta
		 * @param $new_basenames
		 *
		 * @return mixed
		 */

		public function update_meta_sizes_file( $meta, $new_basenames ) {

			if ( is_array( $meta['sizes'] ) ) {
				$i = 0;

				// updates $meta['sizes'] with $new_basenames
				foreach ( $meta['sizes'] as $size => $data ) {
					$meta['sizes'][ $size ]['file'] = $new_basenames[ ++ $i ];
				}
			}

			return $meta;
		}


		/**
		 * update_backup_files
		 *
		 * updates _wp_attachment_backup_sizes values
		 * _wp_attachment_backup_sizes are used for restoring images after it was edited in image editor please se  wp_restore_image() core function.
		 *
		 * @param $attachment_id
		 * @param $backups
		 * @param $new_basenames
		 */

		public function update_backup_files( $attachment_id, $backups, $new_basenames ) {

			// $backups is _wp_attachment_backup_sizes
			if ( is_array( $backups ) ) {

				$i = 0;
				$l = count( $backups );

				// updates vakup sizes with $new_basenames
				$new_backup_sizes = array_slice( $new_basenames, - $l, $l );

				foreach ( $backups as $size => $data ) {
					$backups[ $size ]['file'] = $new_backup_sizes[ $i ++ ];
				}

				// updated post meta _wp_attachment_backup_sizes
				update_post_meta( $attachment_id, '_wp_attachment_backup_sizes', $backups );
			}
		}
	}
}