<?php

// deny direct access
if ( ! function_exists( 'add_action' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit();
}

/**
 * Class WPF_Downloads_Admin
 *
 * Admin class
 */
class WPF_Downloads_Admin {

	/**
	 * Get things started
	 *
	 * @since 1.0
	 * @return void
	 */

	private $wpfd_htacess;


	public function __construct() {

		$this->wpfd_htacess = new WPF_Downloads_htacess();

		add_filter( 'wpf_configure_sections', array( $this, 'configure_sections' ), 10, 2 );
		add_filter( 'wpf_configure_settings', array( $this, 'register_settings' ), 15, 2 );

		add_filter( 'wpf_compatibility_notices', array( $this, 'compatibility_notices' ) );

		// adding our fields to the media library
		add_filter( 'attachment_fields_to_edit', array( $this, 'attachment_fields_to_edit' ), 10, 2 );

		// saving the values of the added fields
		add_filter( 'attachment_fields_to_save', array( $this, 'attachment_fields_to_save' ), 10, 2 );

		//add_action( 'admin_init', array( $this, 'check_htaccess_updated' ) );
		add_filter( 'mod_rewrite_rules', array( $this, 'htaccess_rules' ), 20 );

		// Scripts and styles
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

	}


	/**
	 * Adds Addons tab if not already present
	 *
	 * @access public
	 * @return void
	 */

	public function configure_sections( $page, $options ) {

		if ( ! isset( $page['sections']['addons'] ) ) {
			$page['sections'] = wp_fusion()->settings->insert_setting_before( 'import', $page['sections'], array( 'addons' => __( 'Addons', 'wp-fusion' ) ) );
		}

		return $page;

	}

	/**
	 * Add fields to settings page
	 *
	 * @access public
	 * @return array Settings
	 */

	public function register_settings( $settings, $options ) {

		$settings['downloads_header'] = array(
			'title'   => __( 'Downloads Addon', 'wp-fusion' ),
			'std'     => 0,
			'type'    => 'heading',
			'section' => 'addons',
		);

		$settings['downloads_default_redirect'] = array(
			'title'   => __( 'Default Redirect', 'wp-fusion' ),
			'desc'    => __( 'Default redirect URL for when access is denied to a protected file. Leave blank to return a 404 status.', 'wp-fusion' ),
			'std'     => home_url(),
			'type'    => 'text',
			'section' => 'addons',
		);

		return $settings;

	}

	/**
	 * Compatibility checks
	 *
	 * @access public
	 * @return array Notices
	 */

	public function compatibility_notices( $notices ) {

		// check if htaccess is writable
		$is_htaccess_writable = $this->wpfd_htacess->htaccess_writable();

		if ( true !== $is_htaccess_writable ) {

			$notices['downloads-htaccess'] = __( 'Your .htaccess file is not writable. Please make sure it\'s writeable or deactivate the <strong>WP Fusion - Downloads Addon</strong> plugin.', 'wp-fusion' );

		}

		return $notices;

	}

	/**
	 * enqueue_scripts
	 *
	 * Enqueue admin scripts
	 *
	 * @since 1.0
	 * @return void
	 */

	public function enqueue_scripts() {

		wp_enqueue_style( 'wpf-downloads-admin', WPF_DOWNLOADS_DIR_URL . 'assets/wpf-downloads-admin.css', array(), WPF_DOWNLOADS_VERSION );
		wp_enqueue_script( 'wpf-downloads-admin', WPF_DOWNLOADS_DIR_URL . 'assets/wpf-downloads-admin.js', array( 'jquery', 'media-views', 'wpf-admin' ), WPF_DOWNLOADS_VERSION );

	}


	/**
	 * attachment_fields_to_edit
	 *
	 * Adding the section with the custom fields to the Media library with $form_fields array
	 *
	 * @param $form_fields
	 * @param $post
	 *
	 * @return mixed
	 */

	public function attachment_fields_to_edit( $form_fields, $post ) {

		// heading
		$form_fields['wpf_heading'] = array(
			'label' => '<h2>PROTECT THIS FILE</h2>',
			'input' => 'html',
			'html'  => '&nbsp;',
		);

		$defaults = array(
			'lock_content' => false,
			'allow_tags'   => array(),
		);

		$settings = (array) get_post_meta( $post->ID, 'wpf-settings', true );

		$settings = array_merge( $defaults, $settings );

		// Unprotected checkbox
		$checked = ( $settings['lock_content'] ) ? 'checked' : '';

		$form_fields['wpf_protected'] = array(
			'label' => 'Protected',
			'input' => 'html',
			'html'  => '<input class="wpfd-switch-input" type="checkbox" id="attachments[' . $post->ID . '][wpf_protected]" name="attachments[' . $post->ID . '][wpf_protected]" ' . $checked . ' />' . '<label class="wpfd-switch-paddle" for="attachments[' . $post->ID . '][wpf_protected]"> ' . '<span class="wpfd-switch-active" aria-hidden="true">Yes</span><span class="wpfd-switch-inactive" aria-hidden="true">No</span></label>',
		);

		$args = array(
			'setting'   => $settings['allow_tags'],
			'meta_name' => 'attachments[' . $post->ID . '][wpf_required_tags]',
			'disabled'  => ! $settings['lock_content'],
		);

		ob_start();

		wpf_render_tag_multiselect( $args );

		$html = ob_get_clean();

		$form_fields['wpf_required_tags'] = array(
			'label' => 'Required Tags',
			'input' => 'html',
			'html'  => $html . '<span class="desc">' . __( 'Required Tags (any)', 'wp-fusion' ) . '</span>',

		);

		return $form_fields;
	}


	/**
	 * attachment_fields_to_save
	 *
	 * Saving the post meta if file is protected or not together with the associated role.
	 * Moving the file to protected or un-protected dir according the users action using handle_move_file().
	 *
	 * @param array $post
	 * @param array $attachment
	 *
	 * @return array,string
	 */

	public function attachment_fields_to_save( $post, $attachment ) {

		try {

			$post_data = array(
				'lock_content' => false,
				'allow_tags'   => array(),
			);

			if ( isset( $attachment['wpf_protected'] ) ) {

				$post_data['lock_content'] = true;

			}

			if ( ! empty( $attachment['wpf_required_tags'] ) ) {

				$post_data['allow_tags'] = $attachment['wpf_required_tags'];

			}

			update_post_meta( $post['ID'], 'wpf-settings', $post_data );

			flush_rewrite_rules();

			// move file to protected directory
			$this->handle_move_file( $post['ID'], $post_data );

			return $post;

		} catch ( Exception $e ) {

			// throw the error
			new WP_Error( 'file protection error', $e->getMessage() );
			return $post;
		}
	}


	/**
	 * handle_move_file
	 *
	 * Checking if file is protected or no and acting accordingly (protecting/un-protecting)
	 *
	 * @since 1.0
	 *
	 * @param $post_id
	 */

	public function handle_move_file( $post_id, $post_data ) {

		// instantiating the file handling class
		$handle = new WPF_Downloads_File();

		if ( true == $post_data['lock_content'] ) {

			// protecting the file
			$handle->protect_file( $post_id );

		} else {

			// un-protecting the file
			$handle->un_protect_file( $post_id );
		}
	}


	/**
	 * htaccess_rules
	 *
	 * This code is run every time flush_rewrite_rules are run.
	 * In this plugin on plugin_install() and every time media file is updated
	 *
	 * @since 1.0
	 *
	 * @param $rules
	 *
	 * @return string
	 */

	public function htaccess_rules( $rules ) {

		// checks whether or not file exists, and is a regular file with size greater than zero
		$endpoint = WPF_DOWNLOADS_ENDPOINT;

		$new_rule = PHP_EOL . '# BEGIN WP Fusion Downloads Addon' . PHP_EOL;
		$new_rule .= 'RewriteCond %{REQUEST_FILENAME} -s' . PHP_EOL;

		// protect the files agains the bots
		$new_rule .= 'RewriteCond %{HTTP_USER_AGENT} !facebookexternalhit/[0-9]' . PHP_EOL;
		$new_rule .= 'RewriteCond %{HTTP_USER_AGENT} !Twitterbot/[0-9]' . PHP_EOL;
		$new_rule .= 'RewriteCond %{HTTP_USER_AGENT} !Googlebot/[0-9]' . PHP_EOL;
		$new_rule .= 'RewriteCond %{HTTP_REFERER} !^(.*)/wp-admin/(.*)' . PHP_EOL;

		// protect the file with .htaccess
		$main_index_file    = str_replace( trailingslashit( site_url() ), '', 'index.php' );
		$upload_dir_url     = str_replace( 'https', 'http', wp_upload_dir()['baseurl'] );
		$site_url           = str_replace( 'https', 'http', site_url() );
		$main_upload_dir    = str_replace( trailingslashit( $site_url ), '', $upload_dir_url );
		$direct_access_path = $main_index_file . '?' . $endpoint . '=$1&is_direct_access=true&file_type=$2 [QSA,L]' . PHP_EOL;

		// main re-write rule. This rule is making sure that even protected files are visible for logged in users
		$new_rule .= 'RewriteRule ' . $main_upload_dir . '/_wpfd(\/[A-Za-z0-9_@.\/&+-]+)+\.([A-Za-z0-9_@.\/&+-]+)$ ' . $direct_access_path;

		$new_rule .= '# END WP Fusion Downloads Addon' . PHP_EOL;

		return $new_rule . $rules . PHP_EOL . PHP_EOL;
	}

}

$wpdf_admin = new WPF_Downloads_Admin();
