<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://linksoftwarellc.com
 * @since      2.0.0
 *
 * @package    Wp_Terms_Popup
 * @subpackage Wp_Terms_Popup/admin
 * @author     Link Software LLC <support@linksoftwarellc.com>
 */

class Wp_Terms_Popup_Admin
{
    /**
     * The ID of this plugin.
     *
     * @since    2.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    2.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    2.0.0
     * @param      string    $plugin_name       The name of this plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct($plugin_name, $version)
    {
        $this->plugin_name = $plugin_name;
        $this->version = $version;

        $wptp_installed_version = get_option('wptp_version');

        if ($wptp_installed_version != $this->version) {
            update_option('wptp_version', $this->version);
        }
    }

    /**
     * Register the menu for the admin area.
     *
     * @since    2.0.0
     */
    public function menu()
    {
        add_submenu_page('edit.php?post_type=termpopup', __('WP Terms Popup Settings', $this->plugin_name), __('Settings', $this->plugin_name), 'manage_options', $this->plugin_name.'-settings', [$this, 'settings']);
        add_submenu_page('edit.php?post_type=termpopup', __('WP Terms Popup Designer', $this->plugin_name), __('Designer', $this->plugin_name), 'manage_options', $this->plugin_name.'-designer', [$this, 'designer']);
        add_submenu_page('edit.php?post_type=termpopup', __('WP Terms Popup Collector', $this->plugin_name), __('Collector', $this->plugin_name), 'manage_options', $this->plugin_name.'-collector', [$this, 'collector']);

        do_action('wptp_menu');
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    2.0.0
     */
    public function enqueue_styles()
    {
        wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__).'css/wp-terms-popup-admin.css', [], $this->version, 'all');
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    2.0.0
     */
    public function enqueue_scripts()
    {
        wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__).'js/wp-terms-popup-admin.js', ['jquery'], $this->version, false);
    }

    /**
     * Register the WP Terms Popup post type.
     *
     * @since    2.0.0
     */
    public function post_type()
    {
        $labels = [
            'name' => _x('WP Terms Popup', 'post type general name', $this->plugin_name),
            'singular_name' => _x('Popup', 'post type singular name', $this->plugin_name),
            'menu_name' => _x('WP Terms Popup', 'admin menu', $this->plugin_name),
            'name_admin_bar' => _x('Popup', 'add new on admin bar', $this->plugin_name),
            'add_new' => _x('Add New', 'termpopup', $this->plugin_name),
            'add_new_item' => __('Add New Popup', $this->plugin_name),
            'new_item' => __('New Popup', $this->plugin_name),
            'edit_item' => __('Edit Popup', $this->plugin_name),
            'view_item' => __('View Popup', $this->plugin_name),
            'all_items' => __('All Popups', $this->plugin_name),
            'search_items' => __('Search Popups', $this->plugin_name),
            'not_found' => __('No Popups found.', $this->plugin_name),
            'not_found_in_trash' => __('No Popups found in Trash.', $this->plugin_name),
        ];

        $icon = base64_encode(
            '<svg id="svg" version="1.1" width="20" height="20" viewBox="0 0 400 400" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" ><g id="svgg"><path id="path0" d="M61.460 0.497 C 61.561 0.661,61.229 0.783,60.722 0.768 C 59.442 0.731,58.576 0.956,57.927 1.494 C 57.375 1.952,55.282 2.903,54.685 2.966 C 54.512 2.985,54.242 3.180,54.085 3.401 C 53.928 3.621,53.283 4.068,52.652 4.394 C 52.020 4.720,51.403 5.260,51.281 5.594 C 51.158 5.927,50.375 6.891,49.541 7.735 C 48.706 8.580,48.038 9.390,48.055 9.535 C 48.132 10.183,47.467 11.600,47.087 11.600 C 46.856 11.600,46.785 11.718,46.930 11.863 C 47.075 12.008,47.014 12.342,46.795 12.605 C 46.577 12.869,46.332 13.648,46.252 14.337 C 46.172 15.026,45.857 15.897,45.553 16.272 C 45.075 16.861,45.081 16.906,45.600 16.600 C 46.100 16.306,46.116 16.350,45.700 16.865 C 45.208 17.472,45.159 382.487,45.651 384.618 C 45.755 385.068,46.132 385.853,46.490 386.364 C 46.847 386.874,47.044 387.541,46.927 387.846 C 46.810 388.151,46.924 388.400,47.181 388.400 C 47.437 388.400,47.536 388.580,47.400 388.800 C 47.264 389.020,47.354 389.200,47.600 389.200 C 47.846 389.200,47.936 389.380,47.800 389.600 C 47.664 389.820,47.754 390.000,48.000 390.000 C 48.246 390.000,48.336 390.180,48.200 390.400 C 48.064 390.620,48.154 390.800,48.400 390.800 C 48.646 390.800,48.750 390.957,48.631 391.150 C 48.512 391.342,48.862 391.835,49.408 392.245 C 49.953 392.654,50.400 393.213,50.400 393.486 C 50.400 393.758,50.895 394.296,51.500 394.681 C 52.105 395.065,53.026 395.834,53.547 396.390 C 54.068 396.945,54.373 397.177,54.226 396.905 C 54.048 396.575,54.166 396.489,54.577 396.646 C 54.917 396.777,55.107 397.026,55.000 397.200 C 54.753 397.599,55.948 398.080,56.436 397.778 C 56.636 397.654,56.800 397.754,56.800 398.000 C 56.800 398.246,56.960 398.349,57.155 398.228 C 57.350 398.107,57.723 398.266,57.984 398.581 C 58.245 398.895,58.715 399.054,59.029 398.933 C 59.343 398.813,59.600 398.927,59.600 399.186 C 59.600 399.445,60.005 399.624,60.500 399.584 C 60.995 399.544,61.580 399.531,61.800 399.555 C 63.955 399.791,339.200 399.719,339.200 399.482 C 339.200 399.308,339.425 399.200,339.700 399.244 C 339.975 399.287,340.869 399.025,341.687 398.661 C 342.505 398.298,343.360 398.000,343.587 398.000 C 343.814 398.000,344.000 397.820,344.000 397.600 C 344.000 397.380,344.390 397.200,344.866 397.200 C 345.809 397.200,346.109 396.980,348.800 394.312 C 351.035 392.098,352.000 390.912,352.000 390.381 C 352.000 390.156,352.291 389.528,352.646 388.986 C 353.001 388.444,353.136 387.997,352.946 387.994 C 352.756 387.991,352.868 387.818,353.197 387.611 C 353.525 387.403,353.800 387.001,353.808 386.717 C 353.816 386.433,353.955 385.660,354.116 385.000 C 354.518 383.356,354.680 16.027,354.279 16.275 C 354.103 16.384,353.996 16.052,354.041 15.536 C 354.198 13.772,354.008 12.748,353.573 13.017 C 353.339 13.162,353.180 13.037,353.221 12.740 C 353.360 11.727,353.201 11.152,352.842 11.374 C 352.384 11.657,351.913 10.465,352.261 9.902 C 352.416 9.651,352.334 9.594,352.062 9.762 C 351.768 9.943,351.600 9.739,351.600 9.200 C 351.600 8.734,351.420 8.464,351.200 8.600 C 350.980 8.736,350.800 8.500,350.800 8.075 C 350.800 7.187,347.522 4.000,346.609 4.000 C 346.274 4.000,346.000 3.820,346.000 3.600 C 346.000 3.380,345.730 3.200,345.400 3.200 C 345.070 3.200,344.800 3.041,344.800 2.847 C 344.800 2.557,344.062 2.238,342.800 1.982 C 342.690 1.960,342.035 1.593,341.345 1.166 C 340.655 0.740,339.809 0.499,339.464 0.631 C 339.120 0.763,338.664 0.763,338.450 0.631 C 337.882 0.280,61.243 0.146,61.460 0.497 M349.025 6.100 L 350.200 7.400 348.900 6.225 C 348.185 5.579,347.600 4.994,347.600 4.925 C 347.600 4.616,347.930 4.888,349.025 6.100 M281.302 63.775 C 281.908 63.888,282.492 63.986,282.601 63.992 C 282.709 63.998,283.203 64.355,283.699 64.785 C 284.194 65.216,284.960 65.858,285.400 66.214 C 285.840 66.569,285.930 66.728,285.600 66.568 C 285.270 66.407,285.529 66.844,286.175 67.538 C 286.821 68.232,287.198 68.800,287.013 68.800 C 286.828 68.800,286.951 69.115,287.288 69.500 C 287.625 69.885,287.845 70.538,287.778 70.950 C 287.711 71.363,287.867 72.038,288.124 72.450 C 288.485 73.027,288.478 73.383,288.096 73.995 C 287.823 74.432,287.600 75.252,287.600 75.818 C 287.600 76.394,287.426 76.740,287.205 76.603 C 286.987 76.468,286.903 76.602,287.017 76.899 C 287.242 77.485,284.147 81.064,283.558 80.900 C 283.361 80.845,283.200 80.980,283.200 81.200 C 283.200 81.420,282.829 81.600,282.376 81.600 C 281.923 81.600,281.665 81.781,281.801 82.002 C 282.130 82.534,289.625 82.475,199.477 82.646 C 138.618 82.762,119.623 82.687,119.843 82.331 C 120.029 82.029,119.922 81.944,119.541 82.091 C 118.909 82.333,116.128 81.209,116.065 80.686 C 116.044 80.513,115.366 79.748,114.557 78.986 C 113.749 78.224,113.225 77.600,113.393 77.600 C 113.561 77.600,113.497 77.358,113.252 77.063 C 113.007 76.768,112.925 76.408,113.070 76.263 C 113.215 76.118,113.153 76.000,112.933 76.000 C 112.480 76.000,112.453 75.859,112.296 72.733 C 112.233 71.486,112.302 70.539,112.448 70.630 C 112.700 70.785,113.325 69.127,113.385 68.149 C 113.410 67.720,117.068 64.049,117.564 63.955 C 117.694 63.931,118.415 63.840,119.167 63.755 C 119.919 63.669,120.579 63.554,120.634 63.499 C 120.832 63.301,280.230 63.574,281.302 63.775 M155.000 128.400 C 190.750 128.290,220.000 128.346,220.000 128.525 C 220.000 128.704,220.315 128.807,220.700 128.753 C 221.699 128.615,224.367 129.850,224.648 130.580 C 224.777 130.918,225.025 131.108,225.198 131.001 C 225.371 130.894,225.810 131.390,226.173 132.104 C 226.536 132.817,227.151 133.940,227.539 134.600 C 229.944 138.691,225.943 147.200,221.614 147.200 C 221.166 147.200,220.797 147.425,220.794 147.700 C 220.790 148.033,220.658 148.000,220.400 147.600 C 220.137 147.194,220.010 147.163,220.006 147.504 C 219.997 148.222,87.947 148.041,87.707 147.322 C 87.620 147.060,86.885 146.628,86.074 146.362 C 84.466 145.835,82.818 144.492,82.807 143.700 C 82.803 143.425,82.609 143.200,82.376 143.200 C 82.143 143.200,82.064 143.020,82.200 142.800 C 82.336 142.580,82.237 142.400,81.981 142.400 C 81.724 142.400,81.604 142.167,81.713 141.883 C 81.822 141.599,81.661 141.025,81.356 140.607 C 80.681 139.684,80.630 139.048,81.262 139.438 C 81.520 139.598,81.616 139.549,81.479 139.328 C 81.058 138.647,80.773 136.400,81.107 136.400 C 81.282 136.400,81.463 135.946,81.509 135.392 C 81.654 133.637,85.193 129.301,86.049 129.830 C 86.242 129.950,86.400 129.846,86.400 129.600 C 86.400 129.354,86.580 129.264,86.800 129.400 C 87.020 129.536,87.200 129.437,87.200 129.181 C 87.200 128.924,87.478 128.821,87.821 128.953 C 88.162 129.084,88.340 129.026,88.215 128.824 C 88.090 128.622,88.396 128.399,88.894 128.328 C 89.392 128.257,89.845 128.290,89.900 128.400 C 89.955 128.510,119.250 128.510,155.000 128.400 M246.662 128.300 C 246.560 128.465,260.769 128.600,278.238 128.600 C 295.707 128.600,310.000 128.465,310.000 128.300 C 310.000 128.135,310.191 128.000,310.424 128.000 C 310.657 128.000,310.746 128.163,310.623 128.363 C 310.500 128.562,310.732 128.638,311.140 128.532 C 311.547 128.425,312.176 128.633,312.537 128.994 C 312.898 129.356,313.355 129.552,313.552 129.430 C 313.748 129.308,314.132 129.477,314.404 129.804 C 314.676 130.132,315.119 130.400,315.389 130.400 C 315.894 130.400,317.857 132.709,317.729 133.152 C 317.689 133.289,317.890 133.670,318.175 134.000 C 318.460 134.330,318.717 134.870,318.746 135.200 C 318.973 137.808,318.951 139.600,318.692 139.600 C 318.523 139.600,318.498 140.050,318.636 140.600 C 318.790 141.216,318.717 141.600,318.443 141.600 C 318.200 141.600,318.000 141.927,318.000 142.327 C 318.000 143.176,315.608 146.000,314.890 146.000 C 314.620 146.000,314.400 146.174,314.400 146.387 C 314.400 146.601,314.132 146.672,313.803 146.546 C 313.460 146.414,313.100 146.595,312.956 146.970 C 312.799 147.380,312.437 147.539,311.986 147.396 C 311.591 147.270,311.152 147.354,311.011 147.583 C 310.561 148.310,245.799 148.122,245.193 147.391 C 244.927 147.071,244.568 146.896,244.396 147.002 C 243.701 147.432,240.800 145.337,240.800 144.405 C 240.800 144.156,240.665 144.035,240.500 144.135 C 240.335 144.235,239.854 143.856,239.432 143.293 C 238.968 142.674,238.889 142.401,239.232 142.603 C 239.578 142.807,239.527 142.625,239.099 142.136 C 238.714 141.696,238.519 141.215,238.666 141.068 C 238.813 140.920,238.751 140.800,238.528 140.800 C 238.056 140.800,238.056 136.843,238.529 136.100 C 238.704 135.825,238.735 135.419,238.598 135.197 C 238.461 134.975,238.632 134.511,238.978 134.165 C 239.324 133.819,239.530 133.410,239.435 133.257 C 239.340 133.103,239.624 132.533,240.066 131.989 C 240.508 131.445,240.737 131.270,240.576 131.600 C 240.414 131.930,240.919 131.502,241.697 130.649 C 242.485 129.784,243.308 129.219,243.556 129.373 C 243.800 129.524,244.000 129.462,244.000 129.236 C 244.000 129.010,244.300 128.940,244.666 129.081 C 245.152 129.267,245.242 129.192,245.000 128.800 C 244.759 128.410,244.847 128.332,245.322 128.515 C 245.681 128.652,246.081 128.593,246.211 128.383 C 246.341 128.172,246.537 128.000,246.647 128.000 C 246.757 128.000,246.764 128.135,246.662 128.300 M143.907 172.650 C 144.508 172.793,145.180 173.016,145.400 173.144 C 145.620 173.272,146.022 173.411,146.294 173.452 C 147.218 173.593,149.600 175.263,149.600 175.771 C 149.600 176.049,149.966 176.608,150.414 177.013 C 150.862 177.418,151.138 177.986,151.027 178.275 C 150.916 178.564,150.963 178.800,151.131 178.800 C 151.299 178.800,151.480 179.475,151.534 180.300 C 151.587 181.125,151.719 182.309,151.827 182.930 C 151.944 183.608,151.866 183.964,151.632 183.820 C 151.409 183.682,151.312 183.946,151.407 184.436 C 151.580 185.336,150.901 186.845,149.708 188.210 C 148.793 189.257,146.000 191.377,146.000 191.025 C 146.000 190.876,145.607 191.030,145.126 191.367 C 143.934 192.201,90.294 192.334,89.270 191.505 C 88.908 191.212,88.760 191.189,88.916 191.450 C 89.251 192.010,86.796 191.226,85.910 190.491 C 85.589 190.225,85.193 190.140,85.030 190.303 C 84.867 190.467,84.793 190.338,84.867 190.018 C 84.940 189.698,84.730 189.420,84.400 189.400 C 84.070 189.380,83.845 189.147,83.900 188.882 C 83.955 188.617,83.838 188.445,83.640 188.500 C 83.442 188.555,83.278 188.436,83.276 188.236 C 83.274 188.036,82.993 187.716,82.652 187.525 C 82.286 187.320,82.136 186.912,82.285 186.524 C 82.441 186.115,82.376 185.968,82.110 186.132 C 81.531 186.490,81.146 185.355,81.444 184.168 C 81.577 183.636,81.517 183.200,81.310 183.200 C 81.103 183.200,80.933 182.578,80.933 181.818 C 80.933 181.057,81.083 180.528,81.267 180.641 C 81.450 180.755,81.600 180.459,81.600 179.984 C 81.600 178.790,82.792 176.341,83.780 175.507 C 84.231 175.126,84.817 174.631,85.083 174.407 C 85.349 174.183,85.689 174.000,85.839 174.000 C 85.989 174.000,86.336 173.693,86.610 173.318 C 86.943 172.863,87.657 172.639,88.755 172.646 C 89.660 172.651,90.400 172.553,90.400 172.428 C 90.400 172.111,142.560 172.328,143.907 172.650 M170.962 172.462 C 170.795 172.732,170.851 172.816,171.097 172.664 C 171.940 172.143,311.166 172.352,312.118 172.876 C 312.603 173.143,313.222 173.399,313.494 173.445 C 314.665 173.641,316.737 175.294,316.969 176.217 C 317.107 176.769,317.559 177.559,317.974 177.974 C 318.773 178.773,319.512 184.800,318.811 184.800 C 318.620 184.800,318.349 185.323,318.208 185.963 C 317.698 188.284,315.581 190.075,311.600 191.554 C 309.758 192.239,169.554 191.996,168.284 191.307 C 167.578 190.923,166.812 190.604,166.583 190.597 C 165.720 190.574,163.095 187.927,162.949 186.934 C 162.896 186.574,162.702 186.172,162.518 186.040 C 161.661 185.426,161.603 178.897,162.449 178.361 C 162.752 178.169,162.834 178.009,162.630 178.006 C 162.427 178.003,162.654 177.582,163.134 177.070 C 163.614 176.559,163.886 176.019,163.737 175.870 C 163.588 175.722,163.650 175.600,163.875 175.600 C 164.099 175.600,164.731 175.195,165.277 174.700 C 166.823 173.301,168.664 172.373,169.364 172.642 C 169.714 172.776,170.000 172.686,170.000 172.443 C 170.000 172.199,170.281 172.000,170.624 172.000 C 170.993 172.000,171.131 172.188,170.962 172.462 M243.086 216.872 C 243.794 216.899,244.290 217.055,244.189 217.219 C 244.087 217.382,244.283 217.623,244.623 217.754 C 245.043 217.915,245.146 217.832,244.944 217.495 C 244.561 216.858,246.088 218.025,246.520 218.700 C 246.696 218.975,247.101 219.200,247.420 219.200 C 247.739 219.200,248.000 219.430,248.000 219.711 C 248.000 219.992,248.368 220.537,248.817 220.922 C 249.266 221.307,249.521 221.804,249.383 222.028 C 249.244 222.252,249.349 222.697,249.615 223.018 C 249.880 223.338,249.964 223.600,249.800 223.600 C 249.636 223.600,249.682 223.825,249.901 224.100 C 250.390 224.712,250.392 229.200,249.903 229.200 C 249.711 229.200,249.664 229.380,249.800 229.600 C 249.936 229.820,249.846 230.000,249.600 230.000 C 249.354 230.000,249.264 230.180,249.400 230.400 C 249.536 230.620,249.478 230.800,249.272 230.800 C 249.066 230.800,248.676 231.334,248.406 231.987 C 248.135 232.639,247.590 233.276,247.194 233.402 C 246.798 233.528,246.379 233.877,246.264 234.178 C 246.149 234.478,245.502 234.846,244.827 234.994 C 244.152 235.142,243.600 235.429,243.600 235.632 C 243.600 235.834,243.409 236.000,243.176 236.000 C 242.943 236.000,242.888 235.775,243.053 235.500 C 243.292 235.103,243.226 235.103,242.733 235.502 C 241.982 236.110,87.804 236.242,88.179 235.633 C 88.301 235.436,88.043 235.368,87.605 235.483 C 87.101 235.615,86.719 235.455,86.563 235.048 C 86.427 234.694,86.155 234.504,85.958 234.626 C 85.761 234.748,85.600 234.657,85.600 234.424 C 85.600 234.191,85.360 234.000,85.066 234.000 C 83.873 234.000,80.935 229.667,81.366 228.543 C 81.490 228.222,81.420 228.064,81.210 228.194 C 80.582 228.582,81.080 223.077,81.716 222.609 C 82.021 222.384,82.198 221.987,82.111 221.727 C 82.023 221.467,82.322 221.056,82.776 220.813 C 83.229 220.570,83.600 220.136,83.600 219.848 C 83.600 219.560,83.960 218.998,84.400 218.600 C 84.840 218.202,85.200 218.028,85.200 218.213 C 85.200 218.399,85.567 218.157,86.016 217.675 C 86.892 216.735,87.431 216.502,86.957 217.269 C 86.793 217.534,87.020 217.491,87.480 217.169 C 88.149 216.700,231.317 216.427,243.086 216.872 M312.100 216.860 C 312.485 216.854,312.800 217.029,312.800 217.248 C 312.800 217.468,312.979 217.537,313.197 217.402 C 313.675 217.106,314.732 217.642,314.772 218.200 C 314.787 218.420,314.935 218.604,315.100 218.609 C 315.710 218.628,317.600 220.714,317.600 221.369 C 317.600 221.781,317.782 221.935,318.063 221.761 C 318.380 221.565,318.455 221.747,318.300 222.338 C 318.176 222.813,318.238 223.100,318.437 222.977 C 318.637 222.854,318.800 222.839,318.800 222.944 C 318.800 223.049,318.921 223.449,319.068 223.833 C 319.249 224.306,319.186 224.439,318.873 224.245 C 318.593 224.072,318.476 224.166,318.578 224.479 C 318.985 225.735,319.122 228.801,318.780 229.013 C 318.573 229.140,318.205 229.616,317.962 230.071 C 317.625 230.701,317.634 230.826,318.000 230.600 C 318.370 230.372,318.374 230.501,318.020 231.163 C 317.767 231.636,317.389 231.917,317.180 231.788 C 316.971 231.658,316.800 231.793,316.800 232.087 C 316.800 232.707,314.538 234.805,314.216 234.483 C 314.097 234.364,314.000 234.471,314.000 234.721 C 314.000 234.971,313.730 235.071,313.400 234.945 C 313.070 234.818,312.800 234.924,312.800 235.181 C 312.800 235.437,312.632 235.544,312.428 235.417 C 312.223 235.291,311.942 235.370,311.804 235.593 C 311.518 236.056,267.917 236.184,267.456 235.723 C 267.304 235.571,266.825 235.392,266.393 235.326 C 265.960 235.260,265.257 234.890,264.832 234.503 C 263.389 233.193,262.886 232.848,262.696 233.037 C 262.593 233.141,262.282 232.680,262.005 232.013 C 261.729 231.346,261.315 230.800,261.085 230.800 C 260.855 230.800,260.779 230.687,260.917 230.549 C 261.055 230.411,260.930 229.826,260.640 229.249 C 260.349 228.672,260.145 227.803,260.187 227.319 C 260.231 226.818,260.091 226.544,259.864 226.684 C 259.614 226.838,259.579 226.644,259.769 226.166 C 259.936 225.745,260.123 224.982,260.185 224.472 C 260.246 223.961,260.485 223.355,260.717 223.123 C 260.948 222.892,261.048 222.468,260.938 222.182 C 260.828 221.896,261.117 221.459,261.580 221.211 C 262.043 220.963,262.357 220.634,262.278 220.480 C 262.198 220.326,262.733 219.599,263.467 218.865 C 264.200 218.130,264.800 217.639,264.800 217.774 C 264.800 217.908,265.241 217.729,265.780 217.376 C 266.381 216.982,267.117 216.834,267.680 216.993 C 268.193 217.138,268.395 217.108,268.138 216.926 C 267.884 216.747,277.514 216.661,289.538 216.735 C 301.562 216.809,311.715 216.865,312.100 216.860 M312.733 261.204 C 312.807 261.422,313.122 261.600,313.433 261.600 C 313.745 261.600,314.000 261.780,314.000 262.000 C 314.000 262.220,314.360 262.400,314.800 262.400 C 315.240 262.400,315.600 262.627,315.600 262.905 C 315.600 263.183,316.004 263.588,316.498 263.805 C 316.991 264.022,317.460 264.560,317.539 265.000 C 317.618 265.440,317.910 266.070,318.188 266.400 C 318.466 266.730,318.711 267.270,318.733 267.600 C 318.975 271.295,318.973 272.493,318.726 272.646 C 318.566 272.745,318.548 273.119,318.685 273.478 C 318.866 273.950,318.789 274.040,318.406 273.804 C 318.032 273.572,317.971 273.630,318.199 273.999 C 318.397 274.319,318.281 274.698,317.900 274.976 C 317.558 275.226,316.929 276.054,316.503 276.816 C 316.077 277.577,315.475 278.210,315.164 278.223 C 314.854 278.236,313.661 278.731,312.513 279.323 C 310.253 280.490,309.819 280.617,310.200 280.000 C 310.336 279.780,310.257 279.600,310.024 279.600 C 309.791 279.600,309.600 279.735,309.600 279.900 C 309.600 280.268,156.500 280.109,155.903 279.740 C 155.651 279.584,155.593 279.666,155.762 279.938 C 155.919 280.192,155.857 280.400,155.624 280.400 C 155.391 280.400,155.200 280.209,155.200 279.976 C 155.200 279.743,155.020 279.664,154.800 279.800 C 154.580 279.936,154.397 279.812,154.394 279.524 C 154.389 279.100,154.323 279.102,154.050 279.534 C 153.778 279.963,153.663 279.942,153.467 279.430 C 153.332 279.079,153.082 278.816,152.911 278.844 C 152.006 278.992,151.183 278.751,151.400 278.400 C 151.536 278.180,151.399 278.000,151.096 278.000 C 150.422 278.000,148.980 276.397,148.612 275.238 C 148.464 274.771,148.175 274.492,147.971 274.618 C 147.767 274.744,147.578 274.567,147.550 274.224 C 147.494 273.523,147.344 272.850,147.211 272.700 C 146.942 272.396,147.054 268.510,147.327 268.678 C 147.506 268.789,147.620 268.637,147.579 268.340 C 147.392 266.974,147.623 265.967,148.062 266.238 C 148.357 266.421,148.415 266.347,148.222 266.035 C 148.045 265.749,148.226 265.315,148.660 264.986 C 149.067 264.678,149.276 264.420,149.126 264.413 C 148.975 264.406,149.335 263.898,149.926 263.284 C 150.516 262.670,151.255 262.251,151.566 262.352 C 151.881 262.455,152.019 262.355,151.878 262.126 C 151.739 261.901,152.081 261.509,152.638 261.255 C 153.439 260.891,153.591 260.899,153.367 261.297 C 153.148 261.685,153.199 261.692,153.590 261.331 C 153.869 261.073,154.375 260.969,154.714 261.099 C 155.054 261.229,155.229 261.170,155.104 260.968 C 154.966 260.744,185.812 260.641,233.738 260.704 C 285.387 260.771,312.646 260.944,312.733 261.204 M132.048 261.700 C 132.264 262.085,132.658 262.400,132.924 262.400 C 133.191 262.400,133.857 262.959,134.404 263.642 C 134.952 264.325,135.597 265.094,135.837 265.350 C 136.682 266.253,137.345 267.736,136.740 267.371 C 136.288 267.100,136.288 267.154,136.734 267.706 C 137.028 268.068,137.136 268.581,136.973 268.844 C 136.785 269.148,136.846 269.219,137.138 269.038 C 137.392 268.881,137.600 268.933,137.600 269.153 C 137.600 269.373,137.420 269.664,137.200 269.800 C 136.980 269.936,136.800 270.407,136.800 270.847 C 136.800 271.287,136.963 271.546,137.163 271.423 C 137.362 271.300,137.431 271.559,137.316 271.999 C 137.201 272.440,136.948 272.800,136.753 272.800 C 136.559 272.800,136.400 272.991,136.400 273.224 C 136.400 273.457,136.580 273.536,136.800 273.400 C 137.505 272.964,137.228 273.541,136.300 274.443 C 135.805 274.924,135.638 275.182,135.930 275.017 C 136.366 274.769,136.368 274.862,135.945 275.540 C 135.662 275.993,135.347 276.280,135.245 276.179 C 135.143 276.077,134.733 276.490,134.333 277.097 C 133.933 277.704,133.604 278.025,133.603 277.811 C 133.601 277.597,133.311 277.834,132.958 278.338 C 132.606 278.842,132.079 279.162,131.788 279.051 C 131.497 278.939,131.033 279.119,130.757 279.452 C 130.481 279.784,129.928 279.970,129.528 279.865 C 129.128 279.761,128.800 279.793,128.800 279.937 C 128.800 280.450,89.442 280.239,88.895 279.724 C 88.618 279.462,88.028 279.318,87.585 279.403 C 87.066 279.503,86.877 279.399,87.055 279.111 C 87.234 278.822,87.100 278.753,86.677 278.915 C 86.258 279.076,85.801 278.852,85.411 278.294 C 85.075 277.815,84.800 277.583,84.800 277.781 C 84.800 277.978,84.350 277.717,83.800 277.200 C 83.250 276.683,82.800 276.081,82.800 275.862 C 82.800 275.642,82.520 275.154,82.179 274.776 C 81.837 274.399,81.528 273.709,81.493 273.245 C 81.458 272.780,81.319 272.400,81.184 272.400 C 80.982 272.400,80.943 268.435,81.141 268.100 C 81.173 268.045,81.155 267.929,81.100 267.842 C 81.045 267.755,81.360 267.274,81.800 266.773 C 82.240 266.272,82.349 266.010,82.042 266.190 C 81.662 266.414,81.721 266.119,82.229 265.259 C 82.638 264.567,83.127 264.000,83.315 264.000 C 83.738 264.000,84.824 262.942,84.841 262.514 C 84.847 262.341,85.111 262.253,85.426 262.318 C 85.742 262.382,86.000 262.237,86.000 261.994 C 86.000 261.751,86.161 261.652,86.358 261.774 C 86.555 261.896,86.818 261.731,86.942 261.408 C 87.318 260.427,131.496 260.713,132.048 261.700 M302.171 319.335 C 302.816 319.678,303.518 319.850,303.733 319.718 C 303.950 319.584,303.996 319.682,303.838 319.938 C 303.657 320.231,303.859 320.400,304.390 320.400 C 304.923 320.400,305.527 320.909,306.053 321.800 C 306.507 322.570,307.041 323.200,307.240 323.200 C 307.438 323.200,307.600 323.661,307.600 324.224 C 307.600 324.903,307.757 325.150,308.068 324.958 C 308.383 324.764,308.452 324.887,308.281 325.334 C 308.140 325.700,308.163 326.000,308.331 326.000 C 308.745 326.000,308.654 329.666,308.222 330.365 C 307.966 330.778,308.023 330.838,308.438 330.593 C 308.747 330.411,308.640 330.684,308.200 331.200 C 307.760 331.716,307.660 331.984,307.979 331.797 C 308.297 331.609,308.150 332.028,307.651 332.728 C 307.153 333.427,306.842 334.000,306.961 334.000 C 307.079 334.000,306.596 334.609,305.888 335.354 C 305.180 336.098,304.465 336.628,304.300 336.530 C 304.135 336.433,304.000 336.556,304.000 336.805 C 304.000 337.333,302.812 338.616,302.775 338.129 C 302.761 337.948,302.582 338.217,302.378 338.726 C 302.174 339.235,301.825 339.539,301.603 339.402 C 301.382 339.265,301.200 339.450,301.200 339.814 C 301.200 340.177,300.846 340.567,300.413 340.681 C 299.981 340.794,299.518 341.230,299.384 341.649 C 299.251 342.068,298.988 342.316,298.800 342.200 C 298.612 342.084,298.343 342.351,298.202 342.794 C 298.061 343.237,297.688 343.600,297.373 343.600 C 297.058 343.600,296.845 343.817,296.900 344.082 C 296.955 344.347,296.730 344.580,296.400 344.600 C 296.070 344.620,295.841 344.842,295.892 345.094 C 295.942 345.345,295.627 345.664,295.192 345.803 C 294.756 345.941,294.400 346.213,294.400 346.406 C 294.400 346.838,292.618 348.800,292.226 348.800 C 292.070 348.800,291.673 349.070,291.343 349.400 C 291.013 349.730,290.878 350.000,291.043 350.000 C 291.209 350.000,290.739 350.540,290.000 351.200 C 289.261 351.860,288.537 352.400,288.392 352.400 C 288.247 352.400,287.919 352.860,287.663 353.423 C 287.406 353.985,287.017 354.334,286.798 354.199 C 286.579 354.064,286.400 354.263,286.400 354.643 C 286.400 355.023,286.279 355.212,286.130 355.063 C 285.982 354.915,285.554 355.260,285.180 355.830 C 284.807 356.400,284.405 356.806,284.289 356.733 C 284.172 356.660,283.655 357.067,283.139 357.637 C 282.622 358.207,281.546 359.392,280.747 360.269 C 279.947 361.146,278.735 362.096,278.052 362.379 C 277.369 362.661,276.897 363.033,277.003 363.205 C 277.247 363.600,276.049 364.078,275.564 363.778 C 275.364 363.654,275.197 363.788,275.194 364.076 C 275.189 364.457,275.082 364.436,274.800 364.000 C 274.538 363.594,274.410 363.563,274.406 363.904 C 274.402 364.247,273.841 364.375,272.641 364.304 C 271.673 364.247,270.983 364.034,271.106 363.832 C 271.232 363.627,271.020 363.583,270.630 363.733 C 270.244 363.881,269.719 363.870,269.464 363.707 C 269.001 363.413,268.779 363.345,267.976 363.250 C 267.743 363.223,267.664 363.020,267.800 362.800 C 267.936 362.580,267.801 362.400,267.500 362.400 C 267.199 362.400,266.604 362.117,266.176 361.770 C 265.749 361.424,265.653 361.272,265.962 361.432 C 266.331 361.624,266.426 361.565,266.238 361.262 C 266.081 361.008,265.683 360.800,265.353 360.800 C 265.023 360.800,264.847 360.647,264.963 360.460 C 265.078 360.273,264.709 359.872,264.142 359.569 C 263.575 359.265,262.954 358.605,262.762 358.101 C 262.571 357.598,261.950 357.009,261.383 356.794 C 260.816 356.578,260.463 356.221,260.600 356.001 C 260.736 355.780,260.556 355.600,260.200 355.600 C 259.844 355.600,259.630 355.465,259.724 355.300 C 259.818 355.135,259.468 354.589,258.947 354.086 C 258.426 353.583,257.989 353.313,257.975 353.486 C 257.961 353.659,257.786 353.392,257.585 352.894 C 257.385 352.396,256.742 351.806,256.155 351.583 C 255.569 351.360,255.202 351.064,255.341 350.926 C 255.726 350.541,254.768 349.573,254.293 349.866 C 254.043 350.021,253.997 349.928,254.178 349.635 C 254.371 349.324,254.175 349.066,253.640 348.926 C 253.178 348.806,252.800 348.548,252.800 348.353 C 252.800 348.159,252.080 347.280,251.200 346.400 C 250.320 345.520,249.712 344.800,249.849 344.800 C 249.986 344.800,249.851 344.502,249.549 344.139 C 249.247 343.775,249.060 343.125,249.133 342.695 C 249.205 342.265,249.109 342.009,248.918 342.127 C 248.727 342.245,248.575 341.185,248.580 339.771 C 248.585 338.357,248.747 337.200,248.941 337.200 C 249.135 337.200,249.199 336.840,249.084 336.400 C 248.969 335.960,249.038 335.600,249.237 335.600 C 249.437 335.600,249.600 335.319,249.600 334.976 C 249.600 334.633,249.780 334.464,250.000 334.600 C 250.220 334.736,250.400 334.477,250.400 334.024 C 250.400 333.571,250.617 333.245,250.882 333.300 C 251.147 333.355,251.372 333.162,251.382 332.870 C 251.409 332.070,254.329 330.456,255.295 330.709 C 255.894 330.866,256.038 330.786,255.813 330.421 C 255.471 329.867,258.661 329.973,260.200 330.567 C 260.836 330.812,260.877 330.787,260.400 330.444 C 260.070 330.206,260.003 330.009,260.251 330.006 C 260.499 330.003,260.924 330.268,261.196 330.596 C 261.468 330.923,261.850 331.093,262.045 330.972 C 262.240 330.851,262.400 330.954,262.400 331.200 C 262.400 331.446,262.582 331.535,262.804 331.398 C 263.026 331.260,265.312 333.230,267.885 335.774 C 271.695 339.542,274.957 341.873,273.800 340.000 C 273.664 339.780,273.700 339.600,273.881 339.600 C 274.061 339.600,274.927 338.953,275.804 338.162 C 276.682 337.371,277.175 336.841,276.900 336.985 C 275.839 337.541,276.473 336.858,278.000 335.800 C 278.880 335.190,279.600 334.446,279.600 334.146 C 279.600 333.846,279.881 333.600,280.224 333.600 C 280.567 333.600,280.735 333.419,280.598 333.197 C 280.461 332.976,280.676 332.674,281.075 332.527 C 281.958 332.201,283.383 330.741,283.750 329.786 C 283.898 329.401,284.120 329.186,284.243 329.309 C 284.540 329.607,285.919 328.285,285.713 327.900 C 285.625 327.735,285.737 327.600,285.963 327.600 C 286.560 327.600,288.792 325.325,288.491 325.024 C 288.350 324.883,288.632 324.664,289.117 324.537 C 289.603 324.411,290.000 324.058,290.000 323.753 C 290.000 323.449,290.240 323.200,290.532 323.200 C 290.825 323.200,291.301 322.759,291.590 322.219 C 291.878 321.680,292.314 321.275,292.557 321.319 C 292.801 321.364,292.955 321.175,292.900 320.900 C 292.845 320.625,293.055 320.400,293.367 320.400 C 293.678 320.400,293.999 320.203,294.079 319.962 C 294.512 318.665,300.101 318.232,302.171 319.335 " stroke="none" fill="#9ea3a8" fill-rule="evenodd"></path><path id="path1" d="M130.868 0.300 C 168.415 0.356,229.975 0.356,267.668 0.300 C 305.360 0.244,274.640 0.198,199.400 0.198 C 124.160 0.198,93.320 0.244,130.868 0.300 M338.200 0.400 C 338.336 0.620,338.717 0.800,339.047 0.800 C 339.377 0.800,339.536 0.620,339.400 0.400 C 339.264 0.180,338.883 0.000,338.553 0.000 C 338.223 0.000,338.064 0.180,338.200 0.400 M56.300 1.863 C 55.475 2.206,54.800 2.630,54.800 2.806 C 54.800 2.983,55.520 2.780,56.400 2.354 C 58.777 1.206,58.707 0.865,56.300 1.863 M344.800 2.786 C 344.800 3.014,345.070 3.200,345.400 3.200 C 345.730 3.200,346.000 3.380,346.000 3.600 C 346.000 3.820,346.191 4.000,346.424 4.000 C 347.011 4.000,346.515 3.290,345.578 2.788 C 345.043 2.502,344.800 2.501,344.800 2.786 M347.600 4.925 C 347.600 4.994,348.185 5.579,348.900 6.225 L 350.200 7.400 349.025 6.100 C 347.930 4.888,347.600 4.616,347.600 4.925 M351.630 9.200 C 351.628 9.776,352.757 12.066,352.786 11.546 C 352.794 11.406,352.537 10.686,352.216 9.946 C 351.894 9.206,351.631 8.870,351.630 9.200 M47.273 10.600 C 47.022 11.150,46.948 11.599,47.109 11.597 C 47.269 11.595,47.608 11.145,47.862 10.597 C 48.117 10.049,48.191 9.600,48.027 9.600 C 47.863 9.600,47.523 10.050,47.273 10.600 M45.569 15.100 C 45.553 15.485,45.476 16.131,45.399 16.537 C 45.323 16.942,45.372 17.162,45.508 17.025 C 45.645 16.889,45.794 16.557,45.840 16.288 C 45.885 16.020,46.049 15.485,46.204 15.100 C 46.371 14.685,46.305 14.400,46.043 14.400 C 45.799 14.400,45.586 14.715,45.569 15.100 M354.598 200.200 C 354.598 300.960,354.644 342.120,354.700 291.667 C 354.756 241.214,354.756 158.774,354.700 108.467 C 354.644 58.160,354.598 99.440,354.598 200.200 M117.300 64.088 C 116.805 64.327,116.400 64.661,116.400 64.830 C 116.400 65.000,116.940 64.869,117.600 64.540 C 118.748 63.968,119.140 63.569,118.500 63.627 C 118.335 63.642,117.795 63.849,117.300 64.088 M282.800 64.327 C 282.800 64.505,283.295 65.010,283.900 65.450 C 285.320 66.484,286.443 67.056,285.400 66.214 C 284.960 65.858,284.195 65.216,283.700 64.787 C 283.205 64.357,282.800 64.150,282.800 64.327 M287.071 69.949 C 287.195 70.424,287.439 70.901,287.613 71.008 C 288.087 71.301,287.821 70.061,287.298 69.538 C 286.960 69.200,286.902 69.305,287.071 69.949 M287.800 73.200 C 287.800 73.818,287.935 74.240,288.100 74.138 C 288.265 74.036,288.400 73.614,288.400 73.200 C 288.400 72.786,288.265 72.364,288.100 72.262 C 287.935 72.160,287.800 72.582,287.800 73.200 M287.246 75.738 C 287.221 76.144,286.820 77.010,286.356 77.661 C 285.892 78.313,285.602 78.935,285.711 79.044 C 286.078 79.411,287.662 76.415,287.477 75.704 C 287.309 75.060,287.289 75.063,287.246 75.738 M115.600 80.581 C 115.600 80.681,116.275 81.071,117.100 81.448 C 117.925 81.826,118.735 82.205,118.900 82.291 C 119.065 82.377,119.200 82.365,119.200 82.264 C 119.200 82.077,116.130 80.400,115.788 80.400 C 115.684 80.400,115.600 80.482,115.600 80.581 M119.800 82.400 C 119.664 82.620,119.923 82.800,120.376 82.800 C 120.829 82.800,121.200 82.620,121.200 82.400 C 121.200 82.180,120.941 82.000,120.624 82.000 C 120.307 82.000,119.936 82.180,119.800 82.400 M160.700 82.700 C 182.315 82.757,217.685 82.757,239.300 82.700 C 260.915 82.643,243.230 82.596,200.000 82.596 C 156.770 82.596,139.085 82.643,160.700 82.700 M88.456 128.459 C 87.717 128.672,87.177 128.904,87.256 128.975 C 87.556 129.246,90.000 128.693,90.000 128.355 C 90.000 128.160,89.955 128.016,89.900 128.036 C 89.845 128.056,89.195 128.246,88.456 128.459 M155.000 128.400 L 90.200 128.604 155.100 128.702 C 198.165 128.767,220.000 128.665,220.000 128.400 C 220.000 128.180,219.955 128.044,219.900 128.098 C 219.845 128.152,190.640 128.288,155.000 128.400 M310.000 128.306 C 310.000 128.505,310.225 128.668,310.500 128.670 C 310.775 128.673,311.450 128.801,312.000 128.956 C 312.687 129.150,312.812 129.122,312.400 128.867 C 311.402 128.251,310.000 127.923,310.000 128.306 M221.425 128.841 C 221.551 129.045,222.381 129.582,223.269 130.035 C 224.157 130.488,224.763 130.664,224.616 130.427 C 224.469 130.189,223.640 129.651,222.773 129.233 C 221.906 128.814,221.300 128.638,221.425 128.841 M262.700 128.700 C 271.335 128.760,285.465 128.760,294.100 128.700 C 302.735 128.640,295.670 128.590,278.400 128.590 C 261.130 128.590,254.065 128.640,262.700 128.700 M315.763 131.020 C 316.103 131.360,316.652 132.025,316.982 132.496 C 317.312 132.966,317.668 133.265,317.774 133.159 C 318.012 132.921,315.897 130.400,315.459 130.400 C 315.285 130.400,315.422 130.679,315.763 131.020 M239.837 132.200 C 239.476 132.750,239.337 133.200,239.527 133.200 C 239.716 133.200,240.077 132.750,240.327 132.200 C 240.578 131.650,240.717 131.200,240.637 131.200 C 240.557 131.200,240.197 131.650,239.837 132.200 M82.512 132.900 C 81.624 134.405,81.108 136.104,81.637 135.777 C 81.837 135.654,82.000 135.304,82.000 134.999 C 82.000 134.695,82.378 133.805,82.840 133.023 C 83.301 132.240,83.589 131.600,83.479 131.600 C 83.369 131.600,82.934 132.185,82.512 132.900 M317.878 134.377 C 317.996 134.830,318.252 135.200,318.447 135.200 C 318.986 135.200,318.868 134.554,318.231 134.026 C 317.765 133.639,317.702 133.702,317.878 134.377 M228.059 137.800 C 228.020 138.460,227.896 139.450,227.783 140.000 L 227.577 141.000 228.008 140.014 C 228.244 139.472,228.368 138.482,228.283 137.814 L 228.128 136.600 228.059 137.800 M81.005 139.192 C 80.872 139.407,80.958 139.947,81.196 140.392 C 81.791 141.505,82.059 141.409,81.616 140.241 C 81.415 139.714,81.360 139.174,81.492 139.041 C 81.625 138.909,81.624 138.800,81.490 138.800 C 81.357 138.800,81.138 138.976,81.005 139.192 M238.000 140.200 C 238.000 140.530,238.180 140.800,238.400 140.800 C 238.620 140.800,238.800 140.530,238.800 140.200 C 238.800 139.870,238.620 139.600,238.400 139.600 C 238.180 139.600,238.000 139.870,238.000 140.200 M227.200 142.017 C 227.200 142.718,224.286 145.636,222.800 146.423 C 222.030 146.831,221.631 147.173,221.914 147.182 C 222.702 147.209,224.439 145.971,226.094 144.200 C 227.406 142.797,228.007 141.600,227.400 141.600 C 227.290 141.600,227.200 141.788,227.200 142.017 M317.402 142.794 C 317.263 143.231,316.607 144.131,315.944 144.794 C 315.281 145.457,314.883 146.000,315.059 146.000 C 315.559 146.000,318.000 143.117,318.000 142.527 C 318.000 141.748,317.690 141.887,317.402 142.794 M82.300 143.065 C 82.575 143.176,82.803 143.477,82.807 143.733 C 82.815 144.293,84.336 145.761,85.263 146.102 C 85.627 146.236,85.127 145.548,84.152 144.573 C 83.177 143.598,82.249 142.814,82.089 142.832 C 81.930 142.849,82.025 142.954,82.300 143.065 M241.600 145.584 C 241.600 145.971,244.032 147.301,244.282 147.051 C 244.366 146.968,243.796 146.523,243.017 146.063 C 242.238 145.603,241.600 145.388,241.600 145.584 M87.600 147.353 C 87.600 147.438,88.050 147.593,88.600 147.698 C 89.150 147.803,89.600 147.734,89.600 147.545 C 89.600 147.355,89.150 147.200,88.600 147.200 C 88.050 147.200,87.600 147.269,87.600 147.353 M245.348 147.482 C 245.503 147.637,246.118 147.810,246.715 147.868 C 247.651 147.957,247.690 147.922,247.000 147.614 C 246.026 147.179,244.960 147.093,245.348 147.482 M310.000 147.607 C 309.067 147.912,309.085 147.931,310.272 147.884 C 310.972 147.857,311.456 147.691,311.349 147.517 C 311.241 147.343,311.118 147.218,311.076 147.240 C 311.034 147.262,310.550 147.427,310.000 147.607 M122.700 147.900 C 140.355 147.958,169.245 147.958,186.900 147.900 C 204.555 147.842,190.110 147.795,154.800 147.795 C 119.490 147.795,105.045 147.842,122.700 147.900 M263.100 147.900 C 271.515 147.960,285.285 147.960,293.700 147.900 C 302.115 147.840,295.230 147.790,278.400 147.790 C 261.570 147.790,254.685 147.840,263.100 147.900 M90.400 172.500 C 90.400 172.665,102.415 172.786,117.100 172.768 C 131.785 172.751,143.612 172.658,143.383 172.563 C 142.638 172.252,90.400 172.190,90.400 172.500 M170.000 172.496 C 170.000 172.658,201.758 172.794,240.573 172.798 C 279.388 172.801,311.473 172.907,311.873 173.032 C 312.524 173.237,312.531 173.215,311.935 172.830 C 311.167 172.333,170.000 172.000,170.000 172.496 M87.300 172.812 C 86.805 173.011,86.400 173.245,86.400 173.332 C 86.400 173.628,88.411 173.105,88.630 172.752 C 88.897 172.319,88.480 172.338,87.300 172.812 M167.776 172.980 C 167.027 173.540,167.032 173.552,167.906 173.332 C 168.405 173.206,168.910 172.945,169.030 172.752 C 169.368 172.204,168.636 172.338,167.776 172.980 M146.000 173.292 C 146.000 173.387,146.835 174.064,147.855 174.796 C 148.875 175.529,149.633 175.899,149.540 175.619 C 149.380 175.140,146.000 172.918,146.000 173.292 M313.400 173.454 C 314.472 174.498,316.887 176.062,316.739 175.616 C 316.646 175.337,315.766 174.649,314.785 174.087 C 313.803 173.525,313.180 173.240,313.400 173.454 M150.206 177.312 C 150.592 177.924,151.009 178.324,151.132 178.201 C 151.255 178.078,150.939 177.578,150.430 177.089 L 149.504 176.200 150.206 177.312 M318.028 178.700 C 318.138 179.873,318.746 180.526,318.772 179.500 C 318.787 178.895,318.620 178.400,318.400 178.400 C 318.180 178.400,318.013 178.535,318.028 178.700 M318.612 183.717 C 318.441 184.112,318.413 184.547,318.551 184.684 C 318.957 185.090,319.284 184.520,319.097 183.731 C 318.932 183.033,318.910 183.032,318.612 183.717 M81.206 184.876 C 81.218 185.881,81.987 186.708,81.994 185.724 C 81.997 185.242,81.820 184.736,81.600 184.600 C 81.380 184.464,81.203 184.588,81.206 184.876 M150.800 185.557 C 150.800 185.973,149.950 187.278,148.911 188.457 C 147.872 189.636,147.462 190.240,147.999 189.800 C 150.306 187.910,152.297 184.800,151.200 184.800 C 150.980 184.800,150.800 185.141,150.800 185.557 M317.600 186.721 C 317.600 186.898,316.835 187.823,315.900 188.777 C 314.965 189.731,314.568 190.321,315.017 190.087 C 316.362 189.386,318.616 186.400,317.800 186.400 C 317.690 186.400,317.600 186.544,317.600 186.721 M163.481 188.183 C 163.877 188.724,164.427 189.259,164.704 189.372 C 164.981 189.485,164.812 189.043,164.329 188.389 C 163.845 187.735,163.295 187.200,163.106 187.200 C 162.917 187.200,163.086 187.642,163.481 188.183 M83.402 188.802 C 83.802 189.244,84.211 189.522,84.312 189.421 C 84.528 189.205,83.388 188.000,82.967 188.000 C 82.807 188.000,83.003 188.361,83.402 188.802 M86.061 190.618 C 86.424 190.989,86.940 191.220,87.206 191.131 C 87.473 191.042,87.175 190.739,86.545 190.456 C 85.473 189.976,85.442 189.986,86.061 190.618 M144.600 191.085 C 143.541 191.340,142.851 192.000,143.644 192.000 C 143.975 192.000,144.596 191.736,145.024 191.413 C 145.451 191.090,145.710 190.841,145.600 190.859 C 145.490 190.878,145.040 190.979,144.600 191.085 M168.400 191.383 C 169.486 192.022,310.064 192.329,311.062 191.694 C 311.316 191.532,279.636 191.400,240.662 191.400 C 201.688 191.400,169.260 191.264,168.600 191.097 L 167.400 190.793 168.400 191.383 M102.929 191.900 C 110.370 191.961,122.430 191.961,129.729 191.900 C 137.028 191.839,130.940 191.789,116.200 191.789 C 101.460 191.789,95.488 191.839,102.929 191.900 M88.303 216.630 C 88.101 216.833,87.186 217.079,86.300 217.169 C 86.135 217.186,86.000 217.391,86.000 217.624 C 86.000 217.863,86.218 217.911,86.500 217.734 C 87.001 217.421,87.196 217.374,88.500 217.261 C 88.885 217.227,89.200 217.020,89.200 216.800 C 89.200 216.364,88.670 216.263,88.303 216.630 M266.500 216.996 C 264.587 217.828,264.916 218.104,266.900 217.332 C 268.876 216.563,269.145 216.394,268.367 216.416 C 268.055 216.424,267.215 216.686,266.500 216.996 M84.264 218.729 C 83.305 219.789,83.446 220.154,84.433 219.167 C 84.892 218.708,85.191 218.257,85.098 218.164 C 85.005 218.071,84.630 218.325,84.264 218.729 M318.230 222.600 C 318.379 223.260,318.508 224.031,318.517 224.313 C 318.526 224.599,318.686 224.674,318.877 224.483 C 319.068 224.292,318.941 223.533,318.590 222.770 C 317.981 221.445,317.969 221.439,318.230 222.600 M81.463 222.700 C 81.069 223.342,80.782 226.178,81.150 225.793 C 81.357 225.577,81.612 224.725,81.719 223.900 C 81.908 222.435,81.835 222.094,81.463 222.700 M260.414 223.574 C 259.907 224.521,259.884 224.919,260.352 224.630 C 260.705 224.412,261.255 222.800,260.977 222.800 C 260.895 222.800,260.642 223.148,260.414 223.574 M250.093 226.400 C 250.093 227.170,250.175 227.485,250.276 227.100 C 250.376 226.715,250.376 226.085,250.276 225.700 C 250.175 225.315,250.093 225.630,250.093 226.400 M259.842 225.834 C 259.708 226.183,259.751 226.741,259.937 227.074 C 260.182 227.512,260.286 227.337,260.311 226.439 C 260.350 225.064,260.208 224.880,259.842 225.834 M80.809 227.700 C 80.836 229.240,83.913 234.000,84.882 234.000 C 85.075 234.000,84.756 233.505,84.174 232.900 C 83.006 231.685,81.382 228.484,81.536 227.700 C 81.591 227.425,81.447 227.200,81.218 227.200 C 80.988 227.200,80.804 227.425,80.809 227.700 M318.584 227.627 C 318.439 227.861,318.426 228.160,318.556 228.290 C 318.687 228.420,318.570 228.813,318.297 229.163 C 317.862 229.721,317.884 229.735,318.473 229.273 C 318.843 228.984,319.120 228.399,319.088 227.973 C 319.021 227.085,318.949 227.036,318.584 227.627 M86.635 235.156 C 86.874 235.416,87.774 235.738,88.635 235.871 C 90.311 236.130,90.446 235.775,88.791 235.460 C 88.246 235.356,87.440 235.139,87.000 234.977 C 86.396 234.755,86.307 234.799,86.635 235.156 M166.400 235.603 L 90.200 235.805 166.900 235.899 C 209.085 235.950,243.600 235.844,243.600 235.663 C 243.600 235.482,243.375 235.348,243.100 235.367 C 242.825 235.385,208.310 235.491,166.400 235.603 M267.462 235.700 C 267.564 235.865,277.526 236.000,289.600 236.000 C 301.674 236.000,311.636 235.865,311.738 235.700 C 311.840 235.535,301.878 235.400,289.600 235.400 C 277.322 235.400,267.360 235.535,267.462 235.700 M130.403 261.137 C 130.405 261.282,130.840 261.601,131.370 261.847 C 132.125 262.197,132.292 262.184,132.139 261.786 C 131.936 261.256,130.398 260.681,130.403 261.137 M152.400 261.400 C 151.712 261.793,151.603 261.976,152.051 261.986 C 152.409 261.994,152.926 261.730,153.200 261.400 C 153.809 260.666,153.684 260.666,152.400 261.400 M310.800 260.953 C 310.800 261.038,311.250 261.193,311.800 261.298 C 312.350 261.403,312.800 261.334,312.800 261.145 C 312.800 260.955,312.350 260.800,311.800 260.800 C 311.250 260.800,310.800 260.869,310.800 260.953 M149.926 263.288 C 149.335 263.899,148.988 264.400,149.156 264.400 C 149.535 264.400,151.399 262.445,151.168 262.289 C 151.076 262.226,150.516 262.676,149.926 263.288 M316.500 264.032 C 316.885 264.256,317.214 264.746,317.232 265.120 C 317.252 265.558,317.337 265.617,317.469 265.287 C 317.709 264.691,316.911 263.594,316.251 263.612 C 316.003 263.619,316.115 263.808,316.500 264.032 M82.264 265.200 C 81.874 265.860,81.643 266.400,81.750 266.400 C 81.858 266.400,82.330 265.860,82.800 265.200 C 83.270 264.540,83.501 264.000,83.313 264.000 C 83.126 264.000,82.654 264.540,82.264 265.200 M148.000 265.303 C 148.000 265.359,147.832 265.993,147.626 266.710 C 147.393 267.521,147.393 267.928,147.626 267.784 C 147.832 267.657,148.000 267.239,148.000 266.856 C 148.000 266.473,148.216 265.944,148.480 265.680 C 148.853 265.307,148.853 265.200,148.480 265.200 C 148.216 265.200,148.000 265.246,148.000 265.303 M135.988 266.378 C 136.633 267.582,137.271 267.792,136.739 266.625 C 136.482 266.061,136.114 265.600,135.922 265.600 C 135.729 265.600,135.759 265.950,135.988 266.378 M81.012 268.517 C 80.841 268.912,80.813 269.347,80.951 269.484 C 81.357 269.890,81.684 269.320,81.497 268.531 C 81.332 267.833,81.310 267.832,81.012 268.517 M136.900 268.969 C 136.900 269.680,137.036 269.924,137.291 269.669 C 137.547 269.413,137.547 269.034,137.291 268.578 C 136.961 267.987,136.900 268.048,136.900 268.969 M80.891 271.300 C 80.937 271.905,81.115 272.400,81.287 272.400 C 81.718 272.400,81.669 271.525,81.198 270.800 C 80.899 270.341,80.827 270.458,80.891 271.300 M81.468 273.506 C 81.594 274.005,81.855 274.510,82.048 274.630 C 82.596 274.968,82.462 274.236,81.820 273.376 C 81.260 272.627,81.248 272.632,81.468 273.506 M147.483 273.996 C 147.597 274.433,147.843 274.697,148.028 274.583 C 148.422 274.339,148.065 273.200,147.595 273.200 C 147.419 273.200,147.369 273.558,147.483 273.996 M318.057 274.057 C 317.775 274.509,317.632 274.966,317.740 275.073 C 317.847 275.181,318.144 274.878,318.400 274.400 C 319.005 273.270,318.724 272.990,318.057 274.057 M82.800 275.930 C 82.800 276.112,83.279 276.697,83.863 277.230 L 84.927 278.200 83.988 276.900 C 83.012 275.548,82.800 275.375,82.800 275.930 M315.430 277.652 C 315.007 278.063,314.849 278.400,315.081 278.400 C 315.543 278.400,316.609 277.265,316.357 277.042 C 316.271 276.966,315.854 277.240,315.430 277.652 M132.000 278.600 C 131.722 278.935,131.679 279.200,131.902 279.200 C 132.122 279.200,132.526 278.930,132.800 278.600 C 133.078 278.265,133.121 278.000,132.898 278.000 C 132.678 278.000,132.274 278.270,132.000 278.600 M89.200 279.900 C 89.200 280.065,98.110 280.200,109.000 280.200 C 119.890 280.200,128.800 280.065,128.800 279.900 C 128.800 279.735,119.890 279.600,109.000 279.600 C 98.110 279.600,89.200 279.735,89.200 279.900 M155.200 279.904 C 155.200 280.067,189.940 280.200,232.400 280.200 C 274.860 280.200,309.600 280.064,309.600 279.898 C 309.600 279.732,274.860 279.599,232.400 279.602 C 189.940 279.605,155.200 279.741,155.200 279.904 M296.413 318.987 C 295.209 319.283,294.159 319.722,294.079 319.962 C 293.999 320.203,293.648 320.400,293.300 320.400 C 292.952 320.400,292.777 320.511,292.912 320.646 C 293.048 320.781,293.976 320.496,294.976 320.012 C 296.409 319.318,297.341 319.160,299.379 319.266 C 300.800 319.340,301.810 319.248,301.623 319.062 C 301.025 318.467,298.666 318.433,296.413 318.987 M304.400 320.655 C 304.730 320.788,305.337 321.280,305.749 321.748 L 306.498 322.600 305.901 321.600 C 305.573 321.050,304.966 320.558,304.552 320.507 C 304.029 320.442,303.983 320.488,304.400 320.655 M255.755 330.288 C 256.042 330.575,258.850 330.472,259.167 330.163 C 259.258 330.073,258.463 330.000,257.400 330.000 C 256.337 330.000,255.596 330.130,255.755 330.288 M252.848 331.462 C 250.608 332.798,250.926 333.032,253.329 331.816 C 254.369 331.290,255.069 330.766,254.885 330.653 C 254.702 330.539,253.785 330.904,252.848 331.462 M262.800 331.306 C 262.800 331.394,263.565 332.074,264.500 332.817 C 265.435 333.560,265.946 333.852,265.636 333.465 C 265.051 332.737,262.800 331.023,262.800 331.306 M305.404 335.196 C 304.774 335.854,304.335 336.463,304.429 336.550 C 304.523 336.637,305.180 336.099,305.888 335.354 C 306.596 334.609,307.035 334.000,306.863 334.000 C 306.691 334.000,306.034 334.538,305.404 335.196 M249.632 334.900 C 249.614 335.285,249.409 335.600,249.176 335.600 C 248.943 335.600,248.878 335.803,249.032 336.051 C 249.388 336.627,250.143 335.382,249.860 334.685 C 249.750 334.413,249.649 334.508,249.632 334.900 M266.400 334.525 C 266.400 334.594,266.985 335.179,267.700 335.825 L 269.000 337.000 267.825 335.700 C 266.730 334.488,266.400 334.216,266.400 334.525 M277.400 336.250 C 276.681 336.862,276.534 337.105,277.033 336.857 C 277.766 336.493,279.141 335.176,278.749 335.214 C 278.667 335.222,278.060 335.688,277.400 336.250 M269.200 337.325 C 269.200 337.394,269.785 337.979,270.500 338.625 L 271.800 339.800 270.625 338.500 C 269.530 337.288,269.200 337.016,269.200 337.325 M248.530 341.079 C 248.568 341.782,248.740 342.217,248.912 342.046 C 249.084 341.875,249.052 341.300,248.842 340.768 C 248.483 339.859,248.464 339.878,248.530 341.079 M292.996 347.604 C 292.366 348.262,291.998 348.800,292.179 348.800 C 292.586 348.800,294.502 346.769,294.293 346.560 C 294.210 346.476,293.626 346.946,292.996 347.604 M289.375 351.285 C 288.729 351.991,288.560 352.312,289.000 351.997 C 290.200 351.138,291.277 350.000,290.890 350.000 C 290.703 350.000,290.021 350.578,289.375 351.285 M284.598 356.002 C 284.198 356.444,283.966 356.899,284.082 357.015 C 284.197 357.130,284.590 356.769,284.955 356.212 C 285.752 354.996,285.593 354.902,284.598 356.002 M262.800 358.357 C 262.800 358.550,263.250 359.003,263.800 359.363 C 264.350 359.724,264.800 359.957,264.800 359.881 C 264.800 359.806,264.350 359.353,263.800 358.875 C 263.250 358.397,262.800 358.163,262.800 358.357 M280.383 360.223 C 279.402 361.225,278.870 361.927,279.200 361.783 C 279.772 361.532,282.670 358.400,282.330 358.400 C 282.240 358.400,281.364 359.220,280.383 360.223 M354.080 385.080 C 353.830 385.894,353.692 386.626,353.773 386.706 C 353.854 386.787,354.136 386.199,354.400 385.400 C 354.664 384.601,354.802 383.869,354.707 383.774 C 354.612 383.678,354.330 384.266,354.080 385.080 M46.041 386.200 C 46.075 387.080,46.674 388.127,46.918 387.732 C 47.059 387.504,46.916 386.976,46.600 386.559 C 46.284 386.142,46.032 385.980,46.041 386.200 M352.935 387.868 C 353.082 388.016,352.918 388.571,352.569 389.103 C 352.221 389.634,352.027 390.161,352.139 390.272 C 352.325 390.459,353.600 388.320,353.600 387.820 C 353.600 387.699,353.390 387.600,353.133 387.600 C 352.877 387.600,352.787 387.721,352.935 387.868 M344.000 397.600 C 344.000 398.101,344.424 398.101,345.200 397.600 C 345.685 397.286,345.628 397.211,344.900 397.206 C 344.405 397.203,344.000 397.380,344.000 397.600 M341.700 398.596 C 340.985 398.907,340.400 399.262,340.400 399.386 C 340.400 399.609,343.186 398.547,343.546 398.187 C 343.901 397.832,342.997 398.033,341.700 398.596 M131.100 399.900 C 168.775 399.956,230.425 399.956,268.100 399.900 C 305.775 399.844,274.950 399.798,199.600 399.798 C 124.250 399.798,93.425 399.844,131.100 399.900 " stroke="none" fill="#9ea3a8" fill-rule="evenodd"></path><path id="path2" d="" stroke="none" fill="#9ea3a8" fill-rule="evenodd"></path><path id="path3" d="" stroke="none" fill="#9ea3a8" fill-rule="evenodd"></path><path id="path4" d="" stroke="none" fill="$9ea3a8" fill-rule="evenodd"></path></g></svg>'
        );

        $args = [
            'labels' => $labels,
            'public' => true,
            'publicly_queryable' => true,
            'show_ui' => true,
            'show_in_menu' => true,
            'query_var' => true,
            'rewrite' => ['slug' => 'termspopup'],
            'capability_type' => 'page',
            'hierarchical' => true,
            'menu_position' => 20,
            'menu_icon' => 'data:image/svg+xml;base64,'.$icon,
            'supports' => ['title', 'editor', 'author', 'revisions'],
        ];

        register_post_type('termpopup', $args);
    }

    /**
     * WP Terms Popup post type edit columns.
     *
     * @since    2.0.0
     */
    public function post_type_edit_columns($columns)
    {
        $columns = [
            'cb' => '<input type="checkbox" />',
            'title' => __('Popup'),
            'shortcodeid' => __('Shortcode'),
            'author' => __('Author'),
            'date' => __('Date'),
        ];

        return $columns;
    }

    /**
     * WP Terms Popup post type manage columns.
     *
     * @since    2.0.0
     */
    public function post_type_manage_columns($column, $post_id)
    {
        global $post;

        switch ($column) {
            case 'shortcodeid':

                $shortcodeid = $post_id;
                printf(__('[wpterms id="%s"]'), $shortcodeid);

                break;

            default:

                break;
        }
    }

    /**
     * WP Terms Popup meta boxes.
     *
     * @since    2.0.0
     */
    public function meta_boxes()
    {
        $screen = get_post_types([
            'public' => true,
        ]);
        unset($screen['termpopup']);

        // Post Type
        add_meta_box(
            'thepopup-meta',
            esc_html__('WP Terms Popup Settings', $this->plugin_name),
            [$this, 'meta_boxes_post_type'],
            'termpopup',
            'normal',
            'high'
        );

        // Post
        add_meta_box(
            'termpopup-setting',
            esc_html__('WP Terms Popup', $this->plugin_name),
            [$this, 'meta_boxes_post'],
            $screen,
            'side',
            'default'
        );
    }

    /**
     * Display post type meta boxes.
     *
     * @since    2.0.0
     */
    public function meta_boxes_post_type($object, $box)
    {
        do_action('wptp_before_meta_boxes_post_type', $object, $box);

        $default_termsopt_agreetxt = get_option('termsopt_agreetxt');
        $default_termsopt_terms_disagreetxt = get_option('termsopt_disagreetxt');
        $default_termsopt_redirecturl = get_option('termsopt_redirecturl');
        $default_termsopt_buttons_always_visible = get_option('termsopt_buttons_always_visible', 0);

        $post_terms_agreetxt = get_post_meta($object->ID, 'terms_agreetxt', true);
        $post_terms_disagreetxt = get_post_meta($object->ID, 'terms_disagreetxt', true);
        $post_terms_redirecturl = get_post_meta($object->ID, 'terms_redirecturl', true);
        $post_terms_buttons_always_visible = get_post_meta($object->ID, 'terms_buttons_always_visible', true);
        $post_terms_age_on = get_post_meta($object->ID, 'terms_age_on', true);
        $post_terms_age_requirement = get_post_meta($object->ID, 'terms_age_requirement', true);
        $post_terms_age_date_format = get_post_meta($object->ID, 'terms_age_date_format', true);

        $meta_terms_agreetxt = '';
        $meta_terms_disagreetxt = '';
        $meta_terms_redirecturl = '';
        $meta_terms_buttons_always_visible = 0;
        $meta_terms_age_on = 0;
        $meta_terms_age_requirement = '';
        $meta_terms_age_date_format = '';

        // Agree Button Text
        if (strlen($post_terms_agreetxt) == 0) {
            $meta_terms_agreetxt = $default_termsopt_agreetxt;
        } else {
            $meta_terms_agreetxt = $post_terms_agreetxt;
        }

        // Decline Button Text
        if (strlen($post_terms_disagreetxt) == 0) {
            $meta_terms_disagreetxt = $default_termsopt_terms_disagreetxt;
        } else {
            $meta_terms_disagreetxt = $post_terms_disagreetxt;
        }

        // Decline URL Redirect
        if (strlen($post_terms_redirecturl) == 0) {
            $meta_terms_redirecturl = $default_termsopt_redirecturl;
        } else {
            $meta_terms_redirecturl = $post_terms_redirecturl;
        }

        // Buttons Always Visible?
        if (strlen($post_terms_buttons_always_visible) == 0) {
            $meta_terms_buttons_always_visible = $default_termsopt_buttons_always_visible;
        } else {
            $meta_terms_buttons_always_visible = $post_terms_buttons_always_visible;
        }

        // Turn Age Verification On?
        if (strlen($post_terms_age_on) != 0) {
            $meta_terms_age_on = $post_terms_age_on;
        }

        // Minimum Age Requirement?
        if (strlen($post_terms_age_requirement) != 0) {
            $meta_terms_age_requirement = $post_terms_age_requirement;
        }

        // Date Format?
        if (strlen($post_terms_age_date_format) != 0) {
            $meta_terms_age_date_format = $post_terms_age_date_format;
        }

        include 'partials/wp-terms-popup-admin-post-type-meta-boxes.php';

        do_action('wptp_after_meta_boxes_post_type', $object, $box);
    }

    /**
     * Save post type meta boxes.
     *
     * @since    2.0.0
     */
    public function meta_boxes_save_post_type($post_id, $post)
    {
        if (!isset($_POST['terms_popupmeta_nonce']) || !wp_verify_nonce(sanitize_text_field($_POST['terms_popupmeta_nonce']), 'post-type-meta-boxes')) {
            return $post_id;
        }

        $post_type = get_post_type_object($post->post_type);

        if (!current_user_can($post_type->cap->edit_post, $post_id)) {
            return $post_id;
        }

        $meta_keys = ['terms_agreetxt', 'terms_disagreetxt', 'terms_redirecturl', 'terms_buttons_always_visible', 'terms_age_on', 'terms_age_requirement', 'terms_age_date_format'];

        foreach ($meta_keys as $meta_key) {
            $meta_value = get_post_meta($post_id, $meta_key, true);
            $new_meta_value = (isset($_POST[$meta_key]) ? sanitize_text_field($_POST[$meta_key]) : '');

            if ($new_meta_value && '' == $meta_value) {
                add_post_meta($post_id, $meta_key, $new_meta_value, true);
            } elseif ($new_meta_value && $new_meta_value != $meta_value) {
                update_post_meta($post_id, $meta_key, $new_meta_value);
            } elseif ('' == $new_meta_value && $meta_value) {
                if ($meta_key == 'terms_buttons_always_visible') {
                    update_post_meta($post_id, $meta_key, 0);
                } else {
                    delete_post_meta($post_id, $meta_key, $meta_value);
                }
            }
        }
    }

    /**
     * Display post meta boxes.
     *
     * @since    2.0.0
     */
    public function meta_boxes_post($object, $box)
    {
        include 'partials/wp-terms-popup-admin-post-meta-boxes.php';
    }

    /**
     * Save post meta boxes.
     *
     * @since    2.0.0
     */
    public function meta_boxes_save_post($post_id, $post)
    {
        if (
            (
                !isset($_POST['terms_enablepopup_nonce'])
                || !wp_verify_nonce(sanitize_text_field($_POST['terms_enablepopup_nonce']), 'post-meta-boxes')
            )
            &&
            (
                !isset($_POST['terms_disablepopup_nonce'])
                || !wp_verify_nonce(sanitize_text_field($_POST['terms_disablepopup_nonce']), 'post-meta-boxes')
            )
        ) {
            return $post_id;
        }

        $post_type = get_post_type_object($post->post_type);

        if (!current_user_can($post_type->cap->edit_post, $post_id)) {
            return $post_id;
        }

        // Enable specific popup
        if (isset($_POST['terms_enablepopup_nonce']) && wp_verify_nonce(sanitize_text_field($_POST['terms_enablepopup_nonce']), 'post-meta-boxes')) {
            $new_enablepop_value = (isset($_POST['terms_enablepop']) ? sanitize_text_field($_POST['terms_enablepop']) : '');
            $enablepop_key = 'terms_enablepop';
            $enablepop_value = get_post_meta($post_id, $enablepop_key, true);

            if ($new_enablepop_value && '' == $enablepop_value) {
                // If a new meta value was added and there was no previous value, add it.
                add_post_meta($post_id, $enablepop_key, $new_enablepop_value, true);
            } elseif ($new_enablepop_value && $new_enablepop_value != $enablepop_value) {
                // If the new meta value does not match the old value, update it.
                update_post_meta($post_id, $enablepop_key, $new_enablepop_value);
            } elseif ('' == $new_enablepop_value && $enablepop_value) {
                // If there is no new meta value but an old value exists, delete it.
                delete_post_meta($post_id, $enablepop_key, $enablepop_value);
            }

            $new_selectedterms_value = (isset($_POST['terms_selectedterms']) ? sanitize_text_field($_POST['terms_selectedterms']) : '');
            $selectedterms_key = 'terms_selectedterms';
            $selectedterms_value = get_post_meta($post_id, $selectedterms_key, true);

            if ($new_selectedterms_value && '' == $selectedterms_value) {
                add_post_meta($post_id, $selectedterms_key, $new_selectedterms_value, true);
            } elseif ($new_selectedterms_value && $new_selectedterms_value != $selectedterms_value) {
                update_post_meta($post_id, $selectedterms_key, $new_selectedterms_value);
            } elseif ('' == $new_selectedterms_value && $selectedterms_value) {
                delete_post_meta($post_id, $selectedterms_key, $selectedterms_value);
            }
        }

        // Disable sitewide popup
        if (isset($_POST['terms_disablepopup_nonce']) && wp_verify_nonce(sanitize_text_field($_POST['terms_disablepopup_nonce']), 'post-meta-boxes')) {
            $new_disablepop_value = (isset($_POST['terms_disablepop']) ? sanitize_text_field($_POST['terms_disablepop']) : '');
            $disablepop_key = 'terms_disablepop';
            $disablepop_value = get_post_meta($post_id, $disablepop_key, true);

            if ($new_disablepop_value && '' == $disablepop_value) {
                // If a new meta value was added and there was no previous value, add it.
                add_post_meta($post_id, $disablepop_key, $new_disablepop_value, true);
            } elseif ($new_disablepop_value && $new_disablepop_value != $disablepop_value) {
                // If the new meta value does not match the old value, update it.
                update_post_meta($post_id, $disablepop_key, $new_disablepop_value);
            } elseif ('' == $new_disablepop_value && $disablepop_value) {
                // If there is no new meta value but an old value exists, delete it.
                delete_post_meta($post_id, $disablepop_key, $disablepop_value);
            }
        }
    }

    /**
     * Settings.
     *
     * @since    2.0.0
     */
    public function settings()
    {
        if ($_POST) {
            check_admin_referer('wptpd_settings');

            $settings = [
                'termsopt_javascript',
                'termsopt_user_visiblity',
                'termsopt_sitewide',
                'termsopt_page',
                'termsopt_agreetxt',
                'termsopt_disagreetxt',
                'termsopt_redirecturl',
                'termsopt_buttons_always_visible',
                'termsopt_opac',
                'termsopt_expiry',
            ];

            foreach ($settings as $setting) {
                if (isset($_POST[$setting])) {
                    if ($setting == 'termsopt_user_visiblity') {
                        update_option($setting, array_map('sanitize_text_field', $_POST[$setting]));
                    } else {
                        update_option($setting, sanitize_text_field($_POST[$setting]));
                    }
                } else {
                    update_option($setting, '');
                }
            }

            delete_option('termsopt_adminenabled');
        }

        include 'partials/wp-terms-popup-admin-settings.php';
    }

    /**
     * Settings tabs.
     *
     * @since   2.0.0
     */
    public function settings_tabs()
    {
        ?>
        <a href="<?php echo add_query_arg(['post_type' => 'termpopup', 'page' => $this->plugin_name.'-settings'], admin_url('edit.php')); ?>" class="nav-tab<?php echo($_GET['page'] == $this->plugin_name.'-settings' ? '  nav-tab-active' : ''); ?>"><?php _e('Settings', $this->plugin_name); ?></a>
		<a href="<?php echo add_query_arg(['post_type' => 'termpopup', 'page' => $this->plugin_name.'-designer'], admin_url('edit.php')); ?>" class="nav-tab<?php echo($_GET['page'] == $this->plugin_name.'-designer' ? '  nav-tab-active' : ''); ?>"><?php _e('Designer', $this->plugin_name); ?></a>
		<a href="<?php echo add_query_arg(['post_type' => 'termpopup', 'page' => $this->plugin_name.'-collector'], admin_url('edit.php')); ?>" class="nav-tab<?php echo($_GET['page'] == $this->plugin_name.'-collector' ? '  nav-tab-active' : ''); ?>"><?php _e('Collector', $this->plugin_name); ?></a>
        <?php
    }

    /**
     * Designer.
     *
     * @since    2.0.0
     */
    public function designer()
    {
        include 'partials/wp-terms-popup-admin-designer.php';
    }

    /**
     * Collector.
     *
     * @since    2.0.0
     */
    public function collector()
    {
        include 'partials/wp-terms-popup-admin-collector.php';
    }
}
