<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
/**
 * Presto Player integration.
 *
 * @since 1.2.0
 */
class WPF_ET_Presto_Player extends WPF_ET_Integrations_Base {

	/**
	 * The slug for WP Fusion's module tracking.
	 *
	 * @since 1.2.0
	 * @var string $slug
	 */

	public $slug = 'presto-player';

	/**
	 * The plugin name for WP Fusion's module tracking.
	 *
	 * @since 1.2.0
	 * @var string $name
	 */
	public $name = 'Presto Player';

	/**
	 * Get things started.
	 *
	 * @since 1.2.0
	 */
	public function init() {
		add_action( 'presto_player_progress', array( $this, 'video_progress' ), 10, 3 );
	}

	/**
	 * Gets the triggers for the integration.
	 *
	 * @access protected
	 *
	 * @since  1.2.0
	 *
	 * @return array The triggers.
	 */
	protected function setup_triggers() {

		$triggers = array(
			'video_play'     => array(
				'name'         => __( 'Video Play', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered when a video is played.', 'wp-fusion-event-tracking' ),
				'post_types'   => array( 'pp_video_block' ),
				'has_single'   => true,
				'has_global'   => true,
				'option_types' => array( 'video' ),
			),
			'video_complete' => array(
				'name'         => __( 'Video Complete', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered when a video is watched to completion.', 'wp-fusion-event-tracking' ),
				'post_types'   => array( 'pp_video_block' ),
				'has_single'   => true,
				'has_global'   => true,
				'option_types' => array( 'video' ),
			),
		);

		return $triggers;
	}

	/**
	 * Track video progression.
	 *
	 * @since  1.2.0
	 * @param integer $video_id The video id.
	 * @param integer $percent The video precentage.
	 * @param integer $visit_time The visit time.
	 */
	public function video_progress( $video_id, $percent, $visit_time ) {

		$user = wpf_get_current_user();

		if ( empty( $user ) ) {
			return;
		}

		$post_id = $this->get_post_id( $video_id );

		if ( 0 === $post_id ) {
			return;
		}

		// Video Play.
		if ( 0 === $percent ) {
			$events = $this->get_events( 'video_play', $post_id );
			if ( ! empty( $events ) ) {
				$args = $this->get_video_vars( $post_id, $video_id );
				foreach ( $events as $event ) {
					$event = $this->replace_tags( $event, $args );
					$this->track_event( $event, $user->user_email );
				}
			}
		}

		// Video Complete.
		if ( 100 === $percent ) {
			$events = $this->get_events( 'video_complete', $post_id );
			if ( ! empty( $events ) ) {
				$args = $this->get_video_vars( $post_id, $video_id );
				foreach ( $events as $event ) {
					$event = $this->replace_tags( $event, $args );
					$this->track_event( $event, $user->user_email );
				}
			}
		}

	}

	/**
	 * Get post ID by video ID.
	 *
	 * @since  1.2.0
	 * @param int $video_id The video id.
	 * @return int The post ID.
	 */
	public function get_post_id( $video_id ) {

		$post_id = wp_cache_get( $video_id, 'wpf_pp_vids' );

		if ( false !== $post_id ) {
			return $post_id;
		}

		global $wpdb;

		$table_name = $wpdb->prefix . 'presto_player_videos';

		$sql     = $wpdb->prepare( "SELECT post_id FROM $table_name WHERE id = %d", absint( $video_id ) );
		$post_id = $wpdb->get_var( $sql );
		$post_id = intval( $post_id );

		wp_cache_add( $video_id, $post_id, 'wpf_pp_vids', HOUR_IN_SECONDS );

		return $post_id;
	}


	/**
	 * Gets the video options.
	 *
	 * @since  1.2.0
	 *
	 * @return array The video options.
	 */
	public function get_video_options() {

		return array(
			'name'    => __( 'Video Play', 'wp-fusion-event-tracking' ),
			'type'    => 'video',
			'options' => array(
				array(
					'meta'        => 'ID',
					'preview'     => 33,
					'placeholder' => __( 'The video ID', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'post_title',
					'preview'     => 'Video Title',
					'placeholder' => __( 'The video title', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'post_status',
					'preview'     => 'published',
					'placeholder' => __( 'The video status', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'post_date',
					'preview'     => gmdate( 'Y-m-d', strtotime( 'yesterday' ) ),
					'placeholder' => __( 'The date the video was created', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'post_modified',
					'preview'     => gmdate( 'Y-m-d', strtotime( '-3 hours' ) ),
					'placeholder' => __( 'The video\'s last modified date', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'src',
					'preview'     => home_url() . '/video.mp4',
					'placeholder' => __( 'The video src', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'video_id',
					'preview'     => 35,
					'placeholder' => __( 'The video ID', 'wp-fusion-event-tracking' ),
				),

				array(
					'meta'        => 'video_poster',
					'preview'     => home_url() . '/poster.png',
					'placeholder' => __( 'The video poster', 'wp-fusion-event-tracking' ),
				),

			),
		);

	}

	/**
	 * Gets the details from the video for merging.
	 *
	 * @since  1.2.0
	 *
	 * @param  int $post_id The post ID.
	 * @param  int $video_id The video ID.
	 * @return array The video variables.
	 */
	public function get_video_vars( $post_id, $video_id = false ) {
		$post = get_post( $post_id, 'ARRAY_A' );
		if ( empty( $post ) ) {
			return array();
		}

		// Post fields.
		$meta_column  = array_column( $this->get_video_options()['options'], 'meta' );
		$video_fields = wp_array_slice_assoc( $post, $meta_column );

		// Custom fields.
		$blocks = parse_blocks( $post['post_content'] );
		foreach ( $blocks as $block ) {
			// inside wrapper block.
			if ( 'presto-player/reusable-edit' === $block['blockName'] && ! empty( $block['innerBlocks'] ) ) {
				foreach ( $block['innerBlocks'] as $inner_block ) {
					// Sometimes one post has multiple videos that's why we passed $video_id to get the correct values.
					if ( false !== $video_id && intval( $inner_block['attrs']['id'] ) !== intval( $video_id ) ) {
						continue;
					}
					if ( ! empty( $inner_block['attrs']['id'] ) ) {
						$video_fields['video_id'] = $inner_block['attrs']['id'];
					}
					if ( ! empty( $inner_block['attrs']['src'] ) ) {
						$video_fields['src'] = $inner_block['attrs']['src'];
					}
					if ( ! empty( $inner_block['attrs']['poster'] ) ) {
						$video_fields['poster'] = $inner_block['attrs']['poster'];
					}
					break;
				}
			}
		}

		return array(
			'video' => $video_fields,
		);

	}

}

new WPF_ET_Presto_Player();
