<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
/**
 * LifterLMS integration.
 *
 * @since 1.0.0
 */
class WPF_ET_LifterLMS extends WPF_ET_Integrations_Base {

	/**
	 * The slug for WP Fusion's module tracking.
	 *
	 * @since 1.0.0
	 * @var string $slug
	 */

	public $slug = 'lifterlms';

	/**
	 * The plugin name for WP Fusion's module tracking.
	 *
	 * @since 1.0.0
	 * @var string $name
	 */
	public $name = 'LifterLMS';

	/**
	 * Get things started.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		add_filter( 'llms_metabox_fields_lifterlms_course_options', array( $this, 'course_lesson_metabox' ) );

		add_action( 'lifterlms_course_completed', array( $this, 'course_lesson_complete' ), 10, 2 );
		add_action( 'lifterlms_lesson_completed', array( $this, 'course_lesson_complete' ), 10, 2 );
		add_action( 'lifterlms_quiz_completed', array( $this, 'quiz_complete' ), 10, 3 );

	}


	/**
	 * Gets the triggers for the integration.
	 *
	 * @access protected
	 *
	 * @since  1.0.0
	 *
	 * @return array The triggers.
	 */
	protected function setup_triggers() {

		$triggers = array(
			'course_progress' => array(
				'name'         => __( 'Course Progress', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered whenever a lesson, topic, or quiz is completed within the course, as well as when the course itself is completed.', 'wp-fusion-event-tracking' ),
				'post_types'   => array( 'course' ),
				'has_single'   => true,
				'has_global'   => true,
				'option_id'    => 'course',
				'option_types' => array( 'course' ),
			),
			'quiz_completed'  => array(
				'name'         => __( 'Quiz Completed', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered whenever a quiz in this course is marked complete.', 'wp-fusion-event-tracking' ),
				'post_types'   => array( 'course' ),
				'has_single'   => true,
				'has_global'   => true,
				'option_id'    => 'quiz',
				'option_types' => array( 'course', 'quiz' ),
			),
		);

		return $triggers;
	}

	/**
	 * Override the add_meta_boxes function in the parent class.
	 *
	 * @since 1.2.0
	 */
	public function add_meta_boxes() {}

	/**
	 * Triggered when course / lesson marked complete
	 *
	 * @since 1.0.0
	 *
	 * @param int $user_id The user ID.
	 * @param int $post_id The post ID.
	 */
	public function course_lesson_complete( $user_id, $post_id ) {

		if ( 'lesson' === get_post_type( $post_id ) ) {
			$lesson    = new LLMS_Lesson( $lesson_id );
			$course_id = $lesson->get_parent_course();
		} else {
			$course_id = $post_id;
		}

		$events = $this->get_events( 'course_progress', $course_id );

		if ( ! empty( $events ) ) {

			$args = $this->get_course_vars( $course_id, $post_id );
			$user = get_user_by( 'id', $user_id );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $user->user_email );
			}
		}
	}


	/**
	 * Triggered when quiz completed.
	 *
	 * @since 1.0.0
	 *
	 * @param int   $user_id The user ID.
	 * @param int   $quiz_id The quiz ID.
	 * @param array $quiz    The quiz data.
	 */
	public function quiz_complete( $user_id, $quiz_id, $quiz ) {

		$lesson_id = $quiz->get( 'lesson_id' );
		$lesson    = new LLMS_Lesson( $lesson_id );
		$course_id = $lesson->get_parent_course();

		$events = $this->get_events( 'course_progress', $course_id );

		if ( ! empty( $events ) ) {

			$args = $this->get_course_vars( $course_id, $post_id );
			$user = get_user_by( 'id', $user_id );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $user->user_email );
			}
		}

		$events = $this->get_events( 'quiz_completed', $course_id );

		if ( ! empty( $events ) ) {

			$course_vars = $this->get_course_vars( $course_id );
			$quiz_vars   = $this->get_quiz_vars( $quiz_id, $quiz );
			$args        = array_merge( $course_vars, $quiz_vars );

			$user = get_user_by( 'id', $user_id );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $user->user_email );
			}
		}

	}


	/**
	 * Add tab in course with data.
	 *
	 * @since  1.0.0
	 *
	 * @param  array $fields The fields.
	 * @return array The fields.
	 */
	public function course_lesson_metabox( $fields ) {

		global $post;

		$fields['wpf-et'] = array(
			'title'  => 'Event Tracking',
			'fields' => array(),
		);

		if ( 'course' === $post->post_type ) {

			$fields['wpf-et']['fields'][] = array(
				'class' => 'course_progress',
				'desc'  => __( 'Triggered whenever a lesson, topic, or quiz is completed within the course, as well as when the course itself is completed.', 'wp-fusion-event-tracking' ),
				'id'    => 'wpf_event_tracking_settings[course_progress]',
				'label' => __( 'Course Progress', 'wp-fusion-event-tracking' ),
				'type'  => 'custom_html',
				'value' => '<br />' . $this->render_event_tracking_fields( 'course_progress', $post->ID, array( 'return' => true ) ),
			);

			$fields['wpf-et']['fields'][] = array(
				'class' => 'quiz_completed',
				'desc'  => __( 'Triggered whenever a quiz in this course is marked complete.', 'wp-fusion-event-tracking' ),
				'id'    => 'wpf_event_tracking_settings[quiz_completed]',
				'label' => __( 'Quiz Completed', 'wp-fusion-event-tracking' ),
				'type'  => 'custom_html',
				'value' => '<br />' . $this->render_event_tracking_fields( 'quiz_completed', $post->ID, array( 'return' => true ) ),
			);

		}

		return $fields;

	}



	/**
	 * Gets the course options.
	 *
	 * @since  1.0.0
	 *
	 * @return array The course options.
	 */
	public function get_course_options() {

		return array(
			'name'    => __( 'Course', 'wp-fusion-event-tracking' ),
			'type'    => 'course',
			'options' => array(
				array(
					'meta'        => 'id',
					'preview'     => 55,
					'placeholder' => __( 'The course ID', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'title',
					'preview'     => 'My cool course',
					'placeholder' => __( 'The course title', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'last_completed_step',
					'preview'     => 'Lesson One',
					'placeholder' => __( 'The title of the last completed course step', 'wp-fusion-event-tracking' ),
				),
			),
		);

	}

	/**
	 * Gets the details from the course for merging.
	 *
	 * @since  1.0.0
	 *
	 * @param  int      $course_id              The course ID.
	 * @param  int|bool $last_completed_step_id The last completed step ID.
	 * @return array    The course variables.
	 */
	public function get_course_vars( $course_id, $last_completed_step_id = false ) {

		$vars = array(
			'course' => array(
				'id'    => $course_id,
				'title' => get_the_title( $course_id ),
			),
		);

		if ( ! empty( $last_completed_step_id ) ) {
			$vars['course']['last_completed_step'] = get_the_title( $last_completed_step_id );
		}

		return $vars;
	}

	/**
	 * Gets the quiz options.
	 *
	 * @since  1.0.0
	 *
	 * @return array The quiz options.
	 */
	public function get_quiz_options() {

		return array(
			'name'    => __( 'Quiz', 'wp-fusion-event-tracking' ),
			'type'    => 'quiz',
			'options' => array(
				array(
					'meta'        => 'id',
					'preview'     => 55,
					'placeholder' => __( 'The quiz ID', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'title',
					'preview'     => 'My cool quiz',
					'placeholder' => __( 'The quiz title', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'grade',
					'preview'     => '85',
					'placeholder' => __( 'The user\'s quiz grade', 'wp-fusion-event-tracking' ),
				),
			),
		);

	}

	/**
	 * Gets the details from the quiz for merging.
	 *
	 * @since  1.0.0
	 *
	 * @param  int       $quiz_id The quiz ID.
	 * @param  LLMS_Quiz $quiz    The quiz result data.
	 * @return array     The quiz variables.
	 */
	public function get_quiz_vars( $quiz_id, $quiz = false ) {

		// Quiz ID is a course ID in the admin :( .
		// @Todo need to make this work better.

		if ( 'course' === get_post_type( $quiz_id ) ) {
			return array();
		}

		$vars = array(
			'quiz' => array(
				'id'    => $quiz_id,
				'title' => get_the_title( $quiz_id ),
			),
		);

		if ( $quiz ) {
			$vars['quiz']['grade'] = $quiz->get( 'grade' );
		}

		return $vars;

	}



}

new WPF_ET_LifterLMS();
