<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
/**
 * LearnDash integration.
 *
 * @since 1.0.0
 */
class WPF_ET_LearnDash extends WPF_ET_Integrations_Base {

	/**
	 * The slug for WP Fusion's module tracking.
	 *
	 * @since 1.0.0
	 * @var string $slug
	 */

	public $slug = 'learndash';

	/**
	 * The plugin name for WP Fusion's module tracking.
	 *
	 * @since 1.0.0
	 * @var string $name
	 */
	public $name = 'LearnDash';

	/**
	 * Get things started.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		add_action( 'learndash_course_completed', array( $this, 'course_completed' ), 5 );
		add_action( 'learndash_lesson_completed', array( $this, 'course_progress' ), 5 );
		add_action( 'learndash_topic_completed', array( $this, 'course_progress' ), 5 );

		add_action( 'learndash_quiz_completed', array( $this, 'quiz_completed' ), 5, 2 );

		add_filter( 'learndash_header_data', array( $this, 'header_data' ) );

	}

	/**
	 * Gets the triggers for the integration.
	 *
	 * @access protected
	 *
	 * @since  1.0.0
	 *
	 * @return array The triggers.
	 */
	protected function setup_triggers() {

		$triggers = array(
			'course_progress' => array(
				'name'         => __( 'Course Progress', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered whenever a lesson or topic is completed within the course, as well as when the course itself is completed.', 'wp-fusion-event-tracking' ),
				'post_types'   => array( 'sfwd-courses' ),
				'has_single'   => true,
				'has_global'   => true,
				'option_types' => array( 'course' ),
			),
			'quiz_completed'  => array(
				'name'         => __( 'Quiz Completed', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered whenever a quiz in this course is completed.', 'wp-fusion-event-tracking' ),
				'post_types'   => array( 'sfwd-courses' ),
				'has_single'   => true,
				'has_global'   => true,
				'option_id'    => 'quiz',
				'option_types' => array( 'course', 'quiz' ),
			),
		);

		return $triggers;
	}

	/**
	 * Track events on course progress.
	 *
	 * @since 1.3.0
	 *
	 * @param array $data   The progress data.
	 */
	public function course_completed( $data ) {

		$events = $this->get_events( 'course_progress', $data['course']->ID );

		if ( ! empty( $events ) ) {

			$args = $this->get_course_vars( $data['course']->ID, $data['course']->ID );
			$user = new WP_User( $data['user']->ID );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $user->user_email );
			}
		}

	}

	/**
	 * Track events on course progress.
	 *
	 * @since 1.0.0
	 *
	 * @param array $data   The progress data.
	 */
	public function course_progress( $data ) {

		if ( ! isset( $data['progress'] ) || ! isset( $data['progress']['last_id'] ) ) {
			return;
		}

		$events = $this->get_events( 'course_progress', $data['course']->ID );

		if ( ! empty( $events ) ) {

			$args = $this->get_course_vars( $data['course']->ID, $data['progress']['last_id'] );
			$user = new WP_User( $data['user']->ID );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $user->user_email );
			}
		}

		// Only do it once.
		remove_action( 'learndash_lesson_completed', array( $this, 'course_progress' ), 5 );
		remove_action( 'learndash_topic_completed', array( $this, 'course_progress' ), 5 );

	}

	/**
	 * Tracks an event when a quiz is completed.
	 *
	 * @since 1.0.0
	 *
	 * @param array   $data   The quiz data.
	 * @param WP_User $user   The user.
	 */
	public function quiz_completed( $data, $user ) {

		if ( isset( $data['quiz']->ID ) ) {
			$quiz_id = $data['quiz']->ID;
		} else {
			// For grading in the admin.
			$quiz_id = $data['quiz'];
		}

		$events = $this->get_events( 'quiz_completed', $data['course']->ID );

		if ( ! empty( $events ) ) {

			$args = $this->get_course_vars( $data['course']->ID, $quiz_id );
			$args = array_merge( $args, $this->get_quiz_vars( $quiz_id, $data ) );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $user->user_email );
			}
		}

	}


	/**
	 * Adds the meta box to the Settings tab.
	 *
	 * @since  1.0.0
	 *
	 * @param  array $header_data    The header data.
	 * @return array The header data.
	 */
	public function header_data( $header_data ) {

		foreach ( $header_data['tabs'] as $tabindex => $tab ) {

			if ( 'wp-fusion-settings' === $tab['id'] ) {
				$header_data['tabs'][ $tabindex ]['metaboxes'][] = 'wpf-event-tracking';
			}
		}

		return $header_data;

	}

	/**
	 * Displays the meta box content.
	 *
	 * @since 1.0.0
	 *
	 * @param WP_Post $post   The download.
	 */
	public function meta_box_callback( $post ) {

		echo '<table class="form-table wpf-event-tracking"><tbody>';

		echo '<tr>';

		echo '<th scope="row">';
		echo '<label for="downloaded_purchased">' . esc_html__( 'Course Progress', 'wp-fusion-event-tracking' ) . ':</label>';
		echo '<span class="description">' . esc_html__( 'Triggered whenever a lesson, topic, or quiz is completed within the course, as well as when the course itself is completed.', 'wp-fusion-event-tracking' ) . '</span>';
		echo '</th>';
		echo '<td>';

			$this->render_event_tracking_fields( 'course_progress', $post->ID );

		echo '</td>';
		echo '</tr>';

		echo '<tr>';

		echo '<th scope="row">';
		echo '<label for="downloaded_purchased">' . esc_html__( 'Quiz Completed', 'wp-fusion-event-tracking' ) . ':</label>';
		echo '<span class="description">' . esc_html__( 'Triggered whenever a quiz in this course is marked complete.', 'wp-fusion-event-tracking' ) . '</span>';
		echo '</th>';
		echo '<td>';

			$this->render_event_tracking_fields( 'quiz_completed', $post->ID );

		echo '</td>';
		echo '</tr>';

		echo '</table>';

	}

	/**
	 * Gets the course options.
	 *
	 * @since  1.0.0
	 *
	 * @return array The course options.
	 */
	public function get_course_options() {

		return array(
			'name'    => learndash_get_custom_label( 'course' ),
			'type'    => 'course',
			'options' => array(
				array(
					'meta'        => 'id',
					'preview'     => 55,
					'placeholder' => __( 'The course ID', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'title',
					'preview'     => 'My cool course',
					'placeholder' => __( 'The course title', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'last_completed_step',
					'preview'     => 'Lesson One',
					'placeholder' => __( 'The title of the last completed course step', 'wp-fusion-event-tracking' ),
				),
			),
		);

	}

	/**
	 * Gets the details from the course for merging.
	 *
	 * @since  1.0.0
	 *
	 * @param  int      $course_id              The course ID.
	 * @param  int|bool $last_completed_step_id The last completed step ID.
	 * @return array    The course variables.
	 */
	public function get_course_vars( $course_id, $last_completed_step_id = false ) {

		$vars = array(
			'course' => array(
				'id'    => $course_id,
				'title' => get_post_field( 'post_title', $course_id, 'raw' ),
			),
		);

		if ( ! empty( $last_completed_step_id ) ) {
			$vars['course']['last_completed_step'] = get_post_field( 'post_title', $last_completed_step_id, 'raw' );
		}

		return $vars;
	}

	/**
	 * Gets the quiz options.
	 *
	 * @since  1.0.0
	 *
	 * @return array The quiz options.
	 */
	public function get_quiz_options() {

		return array(
			'name'    => learndash_get_custom_label( 'quiz' ),
			'type'    => 'quiz',
			'options' => array(
				array(
					'meta'        => 'id',
					'preview'     => 55,
					'placeholder' => __( 'The quiz ID', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'title',
					'preview'     => 'My cool quiz',
					'placeholder' => __( 'The quiz title', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'percentage',
					'preview'     => '85%',
					'placeholder' => __( 'The user\'s quiz percentage', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'points',
					'preview'     => 10,
					'placeholder' => __( 'The user\'s quiz points', 'wp-fusion-event-tracking' ),
				),
			),
		);

	}

	/**
	 * Gets the details from the quiz for merging.
	 *
	 * @since  1.0.0
	 *
	 * @param  int   $quiz_id The quiz ID.
	 * @param  array $data    The quiz result data.
	 * @return array The product variables.
	 */
	public function get_quiz_vars( $quiz_id, $data = false ) {

		$vars = array(
			'quiz' => array(
				'id'    => $quiz_id,
				'title' => get_post_field( 'post_title', $quiz_id, 'raw' ),
			),
		);

		if ( $data ) {
			$vars['quiz']['percentage'] = $data['percentage'] . '%';
			$vars['quiz']['points']     = $data['points'];
		}

		return $vars;

	}



}

new WPF_ET_LearnDash();
