<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
/**
 * Gravity Forms integration.
 *
 * @since 1.0.0
 */
class WPF_ET_Gravity_Forms extends WPF_ET_Integrations_Base {

	/**
	 * The slug for WP Fusion's module tracking.
	 *
	 * @since 1.0.0
	 * @var string $slug
	 */

	public $slug = 'gravity-forms';

	/**
	 * The plugin name for WP Fusion's module tracking.
	 *
	 * @since 1.0.0
	 * @var string $name
	 */
	public $name = 'Gravity Forms';

	/**
	 * Get things started.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		if ( ! method_exists( 'GFForms', 'include_feed_addon_framework' ) ) {
			return;
		}

		add_action( 'gform_after_submission', array( $this, 'after_submission' ), 10, 2 );

		require_once WPF_EVENT_TRACKING_DIR_PATH . 'includes/integrations/gravity-forms/class-gravity-forms-feed.php';

		GFAddOn::register( 'WPF_ET_Gravity_Forms_Feed' );

		new WPF_ET_Gravity_Forms_Feed();

	}

	/**
	 * Gets the triggers for the integration.
	 *
	 * @access protected
	 *
	 * @since  1.0.0
	 *
	 * @return array The triggers.
	 */
	protected function setup_triggers() {

		$triggers = array(
			'form_submitted' => array(
				'name'         => __( 'Form Submitted', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered each time a form is submitted.', 'wp-fusion-event-tracking' ),
				'has_single'   => true,
				'has_global'   => true,
				'option_types' => array( 'form' ),
			),

		);

		return $triggers;
	}

	/**
	 * Gets the instance of the feed addon.
	 *
	 * @since  1.0.0
	 *
	 * @return WPF_ET_Gravity_Forms_Feed The instance.
	 */
	public function get_instance() {
		return WPF_ET_Gravity_Forms_Feed::get_instance();
	}


	/**
	 * Gets all events bound to a particular trigger.
	 *
	 * @since  1.1.0
	 *
	 * @param  string $trigger The trigger.
	 * @return array The events.
	 */
	public function get_single_events( $trigger ) {

		$events = array();

		$feeds = GFAPI::get_feeds( null, null, 'wpf-event-tracking' );

		if ( ! empty( $feeds ) && ! is_wp_error( $feeds ) ) {

			foreach ( $feeds as $feed ) {

				$form = GFAPI::get_form( $feed['form_id'] );

				$events[] = array(
					'title'    => $form['title'],
					'edit_url' => admin_url( 'admin.php?page=gf_edit_forms&view=settings&subview=wpf-event-tracking&id=' . $feed['form_id'] . '&fid=' . $feed['id'] ),
					'setting'  => array(
						'form_submitted' => $feed['meta']['form_submitted'],
					),
				);
			}
		}

		return $events;

	}

	/**
	 * Gets the form options.
	 *
	 * @since  1.1.0
	 *
	 * @return array The form options.
	 */
	public function get_form_options( $options, $form_id = false ) {

		$options = array(
			'name'    => __( 'Form', 'wp-fusion-event-tracking' ),
			'type'    => 'form',
			'options' => array(),
		);

		if ( $form_id ) {
			$form   = GFAPI::get_form( $form_id );
			$fields = $form['fields'];
		} else {
			$fields = null;
		}

		foreach ( GFCommon::get_merge_tags( $fields, null ) as $tag_group ) {

			foreach ( $tag_group['tags'] as $merge_tag ) {

				if ( false !== strpos( $merge_tag['tag'], 'user:' ) || false !== strpos( $merge_tag['tag'], 'ip:' ) ) {
					continue; // the core integration already handles these.
				}

				$id = str_replace( '{', '', $merge_tag['tag'] );
				$id = str_replace( '}', '', $id );

				$options['options'][] = array(
					'meta'        => $id,
					'preview'     => $merge_tag['label'],
					'placeholder' => $merge_tag['label'],
				);
			}
		}

		return $options;

	}

	/**
	 * Process any global events on forms that don't have feeds configured.
	 *
	 * @since 1.1.0
	 *
	 * @param array $entry  The entry.
	 * @param array $form   The form.
	 */
	public function after_submission( $entry, $form ) {

		$events = $this->get_global_events( 'form_submitted' );

		if ( ! empty( $events ) ) {

			$email_address = wpf_get_current_user_email();

			if ( ! $email_address ) {
				// If the person isn't identified, check the form.
				foreach ( $entry as $field ) {
					if ( is_email( $field ) ) {
						$email_address = $field;
						break;
					}
				}
			}

			foreach ( $events as $event ) {

				// Replace the leading "form:" from the global settings.

				$event['name']  = str_replace( 'form:', '', $event['name'] );
				$event['value'] = str_replace( 'form:', '', $event['value'] );

				$event['name']  = GFCommon::replace_variables( $event['name'], $form, $entry, false, false, false, 'text' );
				$event['value'] = GFCommon::replace_variables( $event['value'], $form, $entry, false, false, false, 'text' );

				$this->track_event( $event, $email_address );

			}
		}

	}

}

new WPF_ET_Gravity_Forms();
