<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Gamipress integration.
 *
 * @since 1.2.0
 */
class WPF_ET_GamiPress extends WPF_ET_Integrations_Base {

	/**
	 * The slug for WP Fusion's module tracking.
	 *
	 * @since 1.2.0
	 * @var string $slug
	 */
	public $slug = 'gamipress';

	/**
	 * The plugin name for WP Fusion's module tracking.
	 *
	 * @since 1.2.0
	 * @var string $name
	 */
	public $name = 'GamiPress';

	/**
	 * Get things started
	 *
	 * @access public
	 * @return void
	 */
	public function init() {

		// Points.
		add_action( 'gamipress_update_user_points', array( $this, 'points_updated' ), 10, 8 );

		// Achievements.
		add_action( 'gamipress_award_achievement', array( $this, 'user_complete_achievement' ), 10, 5 );
		add_action( 'gamipress_revoke_achievement_to_user', array( $this, 'user_revoke_achievement' ), 10, 3 );

		// Ranks.
		add_action( 'gamipress_update_user_rank', array( $this, 'update_user_rank' ), 10, 2 );
	}


	/**
	 * Gets the triggers for the integration.
	 *
	 * @access protected
	 *
	 * @since  1.2.0
	 *
	 * @return array The triggers.
	 */
	protected function setup_triggers() {

		$triggers = array(
			'points_earned'       => array(
				'name'         => __( 'Points Earned', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered each time that points are earned.', 'wp-fusion-event-tracking' ),
				'has_global'   => true,
				'option_types' => array( 'points' ),
			),
			'rank_earned'         => array(
				'name'         => __( 'Rank Earned', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered each time a rank is earned.', 'wp-fusion-event-tracking' ),
				'post_types'   => array_keys( GamiPress()->rank_types ),
				'has_single'   => true,
				'has_global'   => true,
				'option_types' => array( 'rank' ),
			),
			'achievement_earned'  => array(
				'name'         => __( 'Achievement Earned', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered each time that an achievement is earned.', 'wp-fusion-event-tracking' ),
				'post_types'   => array_keys( GamiPress()->achievement_types ),
				'has_single'   => true,
				'has_global'   => true,
				'option_types' => array( 'achievement' ),
			),
			'achievement_revoked' => array(
				'name'         => __( 'Achievement Revoked', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered each time that an achievement is revoked.', 'wp-fusion-event-tracking' ),
				'post_types'   => array_keys( GamiPress()->achievement_types ),
				'has_single'   => true,
				'has_global'   => true,
				'option_types' => array( 'achievement' ),
			),
		);

		return $triggers;
	}

	/**
	 * Triggered when a rank has been updated.
	 *
	 * @since 1.2.0
	 *
	 * @param int     $user_id The user ID.
	 * @param WP_Post $rank    The rank.
	 */
	public function update_user_rank( $user_id, $rank ) {
		$user          = get_user_by( 'id', $user_id );
		$email_address = $user->user_email;
		$events        = $this->get_events( 'rank_earned', $rank->ID );

		if ( ! empty( $events ) ) {

			$args = $this->get_rank_vars( $rank->ID );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $email_address );
			}
		}
	}

	/**
	 * Triggered when a user complete an achievement.
	 *
	 * @since 1.2.0
	 *
	 * @param int    $user_id        The user ID.
	 * @param int    $achievement_id The achievement ID.
	 * @param string $trigger        The trigger.
	 * @param int    $site_id        The site ID.
	 * @param array  $args           The args.
	 */
	public function user_complete_achievement( $user_id, $achievement_id, $trigger, $site_id, $args ) {
		// Check if it's an achievement.
		if ( ! get_post_type( $achievement_id ) || get_post_type( $achievement_id ) === 'points-award' ) {
			return;
		}

		$user   = get_user_by( 'id', $user_id );
		$events = $this->get_events( 'achievement_earned', $achievement_id );

		if ( ! empty( $events ) ) {

			$args = $this->get_achievement_vars( $achievement_id );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $$user->user_email );
			}
		}

	}

	/**
	 * Triggered when an achievement has been revoked.
	 *
	 * @since 1.2.0
	 *
	 * @param int $user_id        The user ID.
	 * @param int $achievement_id The achievement ID.
	 * @param int $earning_id     The earning ID.
	 */
	public function user_revoke_achievement( $user_id, $achievement_id, $earning_id ) {
		// Check if it's an achievement.
		if ( ! get_post_type( $achievement_id ) || get_post_type( $achievement_id ) === 'points-award' ) {
			return;
		}

		$user   = get_user_by( 'id', $user_id );
		$events = $this->get_events( 'achievement_revoked', $achievement_id );

		if ( ! empty( $events ) ) {

			$args = $this->get_achievement_vars( $achievement_id );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $user->user_email );
			}
		}

	}

	/**
	 * Triggered when points has been updated.
	 *
	 * @since 1.2.0
	 *
	 * @param int    $user_id        The user ID.
	 * @param int    $new_points     The points added.
	 * @param int    $total_points   The total points.
	 * @param int    $admin_id       The administrator ID.
	 * @param int    $achievement_id The achievement ID.
	 * @param string $points_type    The points type.
	 * @param string $reason         The reason.
	 * @param string $log_type       The log type.
	 */
	public function points_updated( $user_id, $new_points, $total_points, $admin_id, $achievement_id, $points_type, $reason, $log_type ) {
		if ( empty( $points_type ) ) {
			$key = '_gamipress_points';
		} else {
			$key = '_gamipress_' . $points_type;
		}

		$user   = get_user_by( 'id', $user_id );
		$events = $this->get_events( 'points_earned', false );

		if ( ! empty( $events ) ) {

			$args = $this->get_points_vars( $user_id, $key );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $user->user_email );
			}
		}

	}


	/**
	 * Gets the rank options.
	 *
	 * @since  1.2.0
	 *
	 * @return array The rank options.
	 */
	public function get_rank_options() {

		return array(
			'name'    => __( 'Rank', 'wp-fus ion-event-tracking' ),
			'type'    => 'rank',
			'options' => array(
				array(
					'meta'        => 'rank_name',
					'preview'     => 'Rank Name',
					'placeholder' => __( 'The Rank name', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'congratulations_text',
					'preview'     => 'Congratulations!',
					'placeholder' => __( 'The Congratulations Text', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'points_to_unlock',
					'preview'     => 50,
					'placeholder' => __( 'The points to unlock the rank', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'priority',
					'preview'     => 1,
					'placeholder' => __( 'The Rank Priority', 'wp-fusion-event-tracking' ),
				),

			),
		);
	}


	/**
	 * Gets the rank variables.
	 *
	 * @since  1.2.0
	 *
	 * @param  int $rank_id The rank ID.
	 * @return array The rank variables.
	 */
	public function get_rank_vars( $rank_id ) {
		$rank        = get_post( $rank_id );
		$rank_fields = array(
			'rank_name'            => get_the_title( $rank_id ),
			'congratulations_text' => get_post_meta( $rank_id, '_gamipress_congratulations_text', true ),
			'points_to_unlock'     => get_post_meta( $rank_id, '_gamipress_points_to_unlock', true ),
			'priority'             => $rank->menu_order,
		);

		return array(
			'rank' => $rank_fields,
		);
	}

	/**
	 * Gets the achievement options.
	 *
	 * @since  1.2.0
	 *
	 * @return array The achievement options.
	 */
	public function get_achievement_options() {

		return array(
			'name'    => __( 'Achievement', 'wp-fus ion-event-tracking' ),
			'type'    => 'achievement',
			'options' => array(
				array(
					'meta'        => 'achievement_name',
					'preview'     => 'Ach Name',
					'placeholder' => __( 'The achievement name', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'congratulations_text',
					'preview'     => 'Congratulations!',
					'placeholder' => __( 'The Congratulations Text', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'points_awarded',
					'preview'     => 50,
					'placeholder' => __( 'The points awarded', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'earned_by',
					'preview'     => 'admin',
					'placeholder' => __( 'Achievement earned by', 'wp-fusion-event-tracking' ),
				),
			),
		);
	}

	/**
	 * Gets the achievement variables.
	 *
	 * @since  1.2.0
	 *
	 * @param  int $achievement_id The achievement ID.
	 * @return array The achievement variables.
	 */
	public function get_achievement_vars( $achievement_id ) {
		$achievement_fields = array(
			'achievement_name'      => get_the_title( $achievement_id ),
			'congratulations_text'  => get_post_meta( $achievement_id, '_gamipress_congratulations_text', true ),
			'points_awarded'        => get_post_meta( $achievement_id, '_gamipress_points', true ),
			'earned_by'             => get_post_meta( $achievement_id, '_gamipress_earned_by', true ),
			'achievement_image_url' => ( has_post_thumbnail( $achievement_id ) ? get_the_post_thumbnail_url( $achievement_id ) : '' ),
		);

		return array(
			'achievement' => $achievement_fields,
		);
	}



	/**
	 * Gets the points options.
	 *
	 * @since  1.2.0
	 *
	 * @return array The points options.
	 */
	public function get_points_options() {

		return array(
			'name'    => __( 'Points', 'wp-fus ion-event-tracking' ),
			'type'    => 'points',
			'options' => array(
				array(
					'meta'        => 'new_points',
					'preview'     => 5,
					'placeholder' => __( 'The new points added', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'total_points',
					'preview'     => 33,
					'placeholder' => __( 'The total points', 'wp-fusion-event-tracking' ),
				),
			),
		);
	}

	/**
	 * Gets the points variables.
	 *
	 * @since  1.2.0
	 *
	 * @param  int    $user_id The user ID.
	 * @param string $key The user meta key.
	 * @return array The points variables.
	 */
	public function get_points_vars( $user_id, $key ) {
		$points_fields = array(
			'new_points'   => get_user_meta( $user_id, $key . '_new_points', true ),
			'total_points' => get_user_meta( $user_id, $key . '_points', true ),
		);

		return array(
			'points' => $points_fields,
		);
	}



}

new WPF_ET_GamiPress();
