<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Easy Digital Downloads integration.
 *
 * @since 1.0.0
 */
class WPF_ET_EDD_Software_Licensing extends WPF_ET_Integrations_Base {

	/**
	 * The slug for WP Fusion's module tracking.
	 *
	 * @since 1.0.0
	 * @var string $slug
	 */

	public $slug = 'edd-software-licensing';

	/**
	 * The plugin name for WP Fusion's module tracking.
	 *
	 * @since 1.0.0
	 * @var string $name
	 */
	public $name = 'Easy Digital Downloads Software Licensing';

	/**
	 * Get things started.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		add_action( 'edd_sl_activate_license', array( $this, 'activate_license' ), 10, 2 );
		add_action( 'edd_sl_deactivate_license', array( $this, 'deactivate_license' ), 10, 2 );
		add_action( 'edd_sl_download_package_url', array( $this, 'package_download' ), 10, 3 );

		add_action( 'wpf_event_tracking_edd_meta_box', array( $this, 'meta_box_callback' ) );

	}

	/**
	 * Gets the triggers for the integration.
	 *
	 * @access protected
	 *
	 * @since  1.0.0
	 *
	 * @return array The triggers.
	 */
	protected function setup_triggers() {

		$triggers = array(
			'license_activated'   => array(
				'name'         => __( 'License Activated', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered whenever a license key is activated on a site.', 'wp-fusion-event-tracking' ),
				'has_single'   => true,
				'post_types'   => array( 'download' ),
				'has_global'   => true,
				'option_types' => array( 'license', 'download' ),
			),
			'license_deactivated' => array(
				'name'         => __( 'License Deactivated', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered whenever a license key is deactivated on a site.', 'wp-fusion-event-tracking' ),
				'has_single'   => true,
				'post_types'   => array( 'download' ),
				'has_global'   => true,
				'option_types' => array( 'license', 'download' ),
			),
			'installed_update'    => array(
				'name'         => __( 'Installed Update', 'wp-fusion-event-tracking' ),
				'description'  => __( 'Triggered when a plugin update is downloaded onto the remote site.', 'wp-fusion-event-tracking' ),
				'has_single'   => true,
				'post_types'   => array( 'download' ),
				'has_global'   => true,
				'option_types' => array( 'license', 'download' ),
			),
		);

		return $triggers;
	}

	/**
	 * Override the add_meta_boxes function in the parent class.
	 *
	 * @since 1.2.0
	 */
	public function add_meta_boxes() {}

	/**
	 * Triggered when a license is activated.
	 *
	 * @since 1.0.0
	 *
	 * @param int $license_id  The license ID.
	 * @param int $download_id The download ID.
	 */
	public function activate_license( $license_id, $download_id ) {

		$events = $this->get_events( 'license_activated', $download_id );

		if ( ! empty( $events ) ) {

			$license = edd_software_licensing()->get_license( $license_id );

			$license_args  = $this->get_license_vars( $license_id );
			$download_args = wp_fusion_event_tracking()->integration( 'edd' )->get_download_vars( $download_id );

			$args = array_merge( $license_args, $download_args );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $license->customer->email );
			}
		}

	}

	/**
	 * Triggered when a license is deactivated.
	 *
	 * @since 1.0.0
	 *
	 * @param int $license_id  The license ID.
	 * @param int $download_id The download ID.
	 */
	public function deactivate_license( $license_id, $download_id ) {

		$events = $this->get_events( 'license_deactivated', $download_id );

		if ( ! empty( $events ) ) {

			$license = edd_software_licensing()->get_license( $license_id );

			$license_args  = $this->get_license_vars( $license_id );
			$download_args = wp_fusion_event_tracking()->integration( 'edd' )->get_download_vars( $download_id );

			$args = array_merge( $license_args, $download_args );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $license->customer->email );
			}
		}

	}

	/**
	 * Triggered when an update package is delivered via EDDSL.
	 *
	 * The edd_sl_before_package_download hook is probably more appropriate for
	 * this kind of thing but it runs before the request has been verified and
	 * we'd have to duplicate a lot of the EDD checks.
	 *
	 * @since  1.0.0
	 *
	 * @param  string $file_url    The file url.
	 * @param  int    $download_id The download ID.
	 * @param  string $license_key The license key.
	 * @return string The download file URL.
	 */
	public function package_download( $file_url, $download_id, $license_key ) {

		$events = $this->get_events( 'installed_update', $download_id );

		if ( ! empty( $events ) ) {

			$license = edd_software_licensing()->get_license( $license_key );

			$license_args  = $this->get_license_vars( $license->ID );
			$download_args = wp_fusion_event_tracking()->integration( 'edd' )->get_download_vars( $download_id );

			$args = array_merge( $license_args, $download_args );

			foreach ( $events as $event ) {
				$event = $this->replace_tags( $event, $args );
				$this->track_event( $event, $license->customer->email );
			}
		}

		return $file_url;

	}


	/**
	 * Gets the download options.
	 *
	 * @since  1.0.0
	 *
	 * @return array The download options.
	 */
	public function get_license_options() {

		return array(
			'name'    => __( 'License', 'wp-fusion-event-tracking' ),
			'type'    => 'license',
			'options' => array(
				array(
					'meta'        => 'license_key',
					'preview'     => 'b5f9254fd2eb24216f718aff8eb63309',
					'placeholder' => __( 'The license key', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'status',
					'preview'     => 'active',
					'placeholder' => __( 'The license status', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'activation_count',
					'preview'     => '1',
					'placeholder' => __( 'The license activation count', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'expiration',
					'preview'     => gmdate( 'Y-m-d', strtotime( '+1 year' ) ),
					'placeholder' => __( 'The license expiration date', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'site_url',
					'preview'     => 'https://example.com',
					'placeholder' => __( 'The site URL the license was activated/deactivated on', 'wp-fusion-event-tracking' ),
				),
				array(
					'meta'        => 'version_number',
					'preview'     => '1.0.0',
					'placeholder' => __( 'The version number the plugin was updated to', 'wp-fusion-event-tracking' ),
				),
			),
		);

	}

	/**
	 * Gets the details from the license for merging.
	 *
	 * @since  1.0.0
	 *
	 * @param  int $license_id The license ID.
	 * @return array The product variables.
	 */
	public function get_license_vars( $license_id = false ) {

		$license = edd_software_licensing()->get_license( $license_id );

		if ( ! empty( $license->ID ) ) {

			$vars = array(
				'license' => array(
					'license_key'      => $license->license_key,
					'status'           => $license->status,
					'activation_count' => $license->activation_count,
					'expiration'       => $license->expiration,
					'site_url'         => ! empty( $license->sites ) ? $license->sites[0] : false,
				),
			);

			// If we're activating / deactivating a URL, grab it.

			if ( isset( $_REQUEST['url'] ) ) {
				$vars['license']['site_url'] = urldecode( esc_url_raw( wp_unslash( $_REQUEST['url'] ) ) );
			}

			$download = new EDD_SL_Download( $license->download_id );

			$vars['license']['version_number'] = $download->get_version();

		} else {

			// In the admin preview, $license_id is the download ID.

			$download = new EDD_SL_Download( $license_id );

			$vars = array(
				'license' => array(
					'version_number' => $download->get_version(),
				),
			);

		}

		return $vars;
	}


}

new WPF_ET_EDD_Software_Licensing();
