<?php

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class WPF_Event_Tracking_Admin
 *
 * Handles the admin functionality.
 *
 * @since 1.0.0
 */
class WPF_Event_Tracking_Admin {

	/**
	 * Stores any data that needs to be passed to the admin scripts.
	 *
	 * @var localize_data
	 * @since 1.1.3
	 */
	public $localize_data = array();

	/**
	 * Constructs a new instance.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		add_action( 'admin_footer', array( $this, 'admin_scripts' ) );
		add_action( 'admin_menu', array( $this, 'add_settings_submenu' ), 11 );
		add_action( 'admin_init', array( $this, 'save_settings' ) );

		add_filter( 'wpf_configure_sections', array( $this, 'configure_sections' ), 10, 2 );
		add_filter( 'wpf_configure_settings', array( $this, 'register_settings' ), 10, 2 );

	}

	/**
	 * Used by the plugin integrations to pass data to the admin scripts.
	 *
	 * @since 1.1.3
	 *
	 * @param string $integration The integration slug.
	 * @param string $trigger     The trigger ID.
	 * @param array  $options     The options.
	 */
	public function localize( $integration, $trigger, $options ) {

		if ( ! isset( $this->localize_data['triggers'] ) ) {
			$this->localize_data = array( 'triggers' => array() );
		}

		if ( ! isset( $this->localize_data['triggers'][ $integration ] ) ) {
			$this->localize_data['triggers'][ $integration ] = array();
		}

		if ( ! isset( $this->localize_data['triggers'][ $integration ][ $trigger ] ) ) {
			$this->localize_data['triggers'][ $integration ][ $trigger ] = array();
		}

		$this->localize_data['triggers'][ $integration ][ $trigger ]['options'] = $options;

	}

	/**
	 * Saves the global settings.
	 *
	 * @since 1.0.0
	 */
	public function save_settings() {

		if ( ! isset( $_POST['wpf_event_tracking_options'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( $_POST['wpf_event_tracking_options_nonce'], 'wpf_event_tracking_options' ) ) {
			return;
		}

		$data = wpf_clean( wp_unslash( $_POST['wpf_event_tracking_options'] ) );

		if ( ! empty( $data ) ) {

			// Clean up empties.

			foreach ( $data as $integration => $triggers ) {

				foreach ( $triggers as $i => $rule ) {

					if ( empty( $rule['name'] ) && empty( $rule['value'] ) ) {
						unset( $data[ $integration ][ $i ] );
					}
				}
			}
		}

		if ( ! empty( $data ) ) {
			update_option( 'wpf_event_tracking_options', $data );
		} else {
			delete_option( 'wpf_event_tracking_options', $data );
		}

		add_action(
			'admin_notices',
			function() {
				echo '<div id="message" class="updated fade"><p><strong>' . esc_html__( 'Settings saved.', 'wp-fusion-event-tracking' ) . '</strong></p></div>';
			}
		);

	}

	/**
	 * Add Addons tab to the settings.
	 *
	 * @since  1.0.0
	 *
	 * @param  array $page    The settings page.
	 * @param  array $options The options in the database.
	 * @return array The settings page with the new section added.
	 */
	public function configure_sections( $page, $options ) {

		if ( ! isset( $page['sections']['addons'] ) ) {
			$page['sections'] = wp_fusion()->settings->insert_setting_before( 'import', $page['sections'], array( 'addons' => __( 'Addons', 'wp-fusion-event-tracking' ) ) );
		}

		$page['sections'] = wp_fusion()->settings->insert_setting_after(
			'addons',
			$page['sections'],
			array(
				'event_tracking' => array(
					'title' => __( 'Event Tracking', 'wp-fusion' ),
					'slug'  => 'wpf-settings-event-tracking',
					'url'   => admin_url( 'options-general.php?page=wpf-event-tracking' ),
				),
			)
		);

		return $page;

	}


	/**
	 * Add settings to the settings page,
	 *
	 * @since 1.0.0
	 *
	 * @param array $settings The available settings.
	 * @param array $options  The options in the database.
	 * @return array The settings with the new settings added.
	 */
	public function register_settings( $settings, $options ) {

		$settings['event_tracking_header'] = array(
			'title'   => __( 'Event Tracking', 'wp-fusion-event-tracking' ),
			'url'     => 'https://wpfusion.com/documentation/event-tracking/event-tracking-overview/',
			'type'    => 'heading',
			'section' => 'addons',
		);

		$settings['events_logging'] = array(
			'title'   => __( 'Log Events', 'wp-fusion-event-tracking' ),
			'desc'    => __( 'Log tracked events to the WP Fusion activity logs.', 'wp-fusion-event-tracking' ),
			'std'     => 1,
			'type'    => 'checkbox',
			'section' => 'addons',
		);

		return $settings;

	}


	/**
	 * Add event tracking submenu page under settings.
	 *
	 * @return void
	 */
	public function add_settings_submenu() {
		add_submenu_page(
			'options-general.php',
			__( 'Event Tracking', 'wp-fusion' ),
			__( 'Event Tracking', 'wp-fusion' ),
			'manage_options',
			'wpf-event-tracking',
			array( $this, 'submenu_callback' )
		);
	}

	/**
	 * Submenu callback.
	 *
	 * @return void
	 */
	public function submenu_callback() {
		include_once WPF_EVENT_TRACKING_DIR_PATH . '/includes/admin/option-page.php';
	}

	/**
	 * Register scripts and styles.
	 *
	 * @since 1.0.0
	 */
	public function admin_scripts() {

		$screen = get_current_screen();

		if ( 'settings_page_wpf-event-tracking' === $screen->id ) {
			wp_enqueue_script( 'wpf-event-tracking-jquery-repeater', WPF_EVENT_TRACKING_DIR_URL . 'assets/jquery-repeater/jquery.repeater.min.js', array( 'jquery' ), WPF_EVENT_TRACKING_VERSION, true );
		}

		wp_enqueue_script( 'wpf-event-tracking-admin', WPF_EVENT_TRACKING_DIR_URL . 'assets/wpf-event-tracking-admin.js', array( 'jquery', 'jquery-ui-sortable' ), WPF_EVENT_TRACKING_VERSION, true );
		wp_enqueue_style( 'wpf-event-tracking-admin', WPF_EVENT_TRACKING_DIR_URL . 'assets/wpf-event-tracking-admin.css', array(), WPF_EVENT_TRACKING_VERSION );

		if ( 'settings_page_wpf-event-tracking' === $screen->id ) {

			$this->localize_data['triggers'] = array();

			// Load the various integration options for the main settings page.

			foreach ( wp_fusion_event_tracking()->integrations as $slug => $integration ) {

				$this->localize_data['triggers'][ $slug ] = array();

				foreach ( $integration->get_triggers() as $trigger => $trigger_data ) {

					$this->localize_data['triggers'][ $slug ][ $trigger ] = array( 'options' => $integration->get_options( $trigger ) );

				}
			}
		}

		if ( ! empty( $this->localize_data ) ) {

			$this->localize_data['crm'] = wp_fusion()->crm->slug;

			$this->localize_data['crms_notices'] = array(
				'title_limit'=>sprintf(__('%s allows a maximum of {limit} characters for the Event Name.','wp-fusion-event-tracking'),wp_fusion()->crm->name),
			);

			wp_localize_script( 'wpf-event-tracking-admin', 'wpfEventData', $this->localize_data );
		}

	}

}

